<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use App\Models\ProductReview;
use App\Models\QuantityUnit;
use App\Models\Subcategory;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /* =========================
        PRODUCTS LIST
    ========================= */
    public function index()
    {
        $products = Product::with(['category', 'subcategory', 'brand', 'quantityUnit'])
            ->orderBy('id', 'desc')
            ->get();

        return view('admin.products.index', compact('products'));
    }


    /* =========================
        CREATE PRODUCT
    ========================= */
    public function create()
    {
        $categories    = Category::where('status', 'active')->get();
        $subcategories = SubCategory::where('status', 'active')->get();
        $brands        = Brand::where('status', 'active')->get();

        return view('admin.products.form', compact(
            'categories',
            'subcategories',
            'brands'
        ));
    }

    /* =========================
        STORE PRODUCT
    ========================= */
    public function store(Request $request)
    {
        
        $request->validate([
            'category_id'       => 'nullable|exists:categories,id',
            'subcategory_id'    => 'nullable|exists:subcategories,id',
            'brand_id'          => 'nullable|exists:brands,id',
            'quantity_unit_id'  => 'nullable|exists:quantity_units,id',
            'store_price'       => 'nullable|numeric|min:0',
            'customer_price'    => 'nullable|numeric|min:0',
            'store_discount_type'  => 'nullable|in:percentage,amount',
            'store_discount_value' => 'nullable|numeric|min:0',
            'customer_discount_type'  => 'nullable|in:percentage,amount',
            'customer_discount_value' => 'nullable|numeric|min:0',
            'status'            => 'required|in:active,inactive',
            'image'             => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'description' => 'nullable|string|max:2000',
            'colors' => 'nullable|array',
            'colors.*.name' => 'required_with:colors|string|max:50',
            'colors.*.hex'  => ['required_with:colors','regex:/^#[A-Fa-f0-9]{6}$/'],
        ]);


        $exists = Product::where('category_id', $request->category_id)
            ->when($request->subcategory_id, function ($q) use ($request) {
                $q->where('subcategory_id', $request->subcategory_id);
            }, function ($q) {
                $q->whereNull('subcategory_id');
            })
            ->where('brand_id', $request->brand_id)
            ->where('quantity_unit_id', $request->quantity_unit_id)
            ->first();


        if ($exists) {
            return back()
                ->withInput()
                ->withErrors([
                    'product' => 'Product already exists with the same category, subcategory, brand, and quantity unit.'
                ]);
        }


        $imagePath = null;

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = 'product_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'uploads/products';

            if (!file_exists(public_path($uploadPath))) {
                mkdir(public_path($uploadPath), 0755, true);
            }

            $file->move(public_path($uploadPath), $filename);
            $imagePath = $uploadPath . '/' . $filename;
        }

        $colors = $request->colors ? array_values($request->colors) : null;

        $quantityUnitName = $request->quantity_unit_id
                            ? optional(QuantityUnit::find($request->quantity_unit_id))->name
                            : null;

        Product::create([
            'category_id'                 => $request->category_id,
            'subcategory_id'              => $request->subcategory_id,
            'brand_id'                    => $request->brand_id,
            'category'                    => optional(Category::find($request->category_id))->name,
            'subcategory'                 => optional(SubCategory::find($request->subcategory_id))->name,
            'brand'                       => optional(Brand::find($request->brand_id))->name,
            'quantity_unit_id'            => $request->quantity_unit_id,
            'quantity_unit'               => $quantityUnitName,
            'store_price'                 => $request->store_price,
            'store_discount_type'         => $request->store_discount_type,
            'store_discount_value'        => $request->store_discount_value,
            'store_price_assigned_by'     => auth()->id(),
            'store_price_assigned_at'     => now(),
            'customer_price'              => $request->customer_price,
            'customer_discount_type'      => $request->customer_discount_type,
            'customer_discount_value'     => $request->customer_discount_value,
            'customer_price_assigned_by'  => auth()->id(),
            'customer_price_assigned_at'  => now(),
            'status'                      => $request->status,
            'image'                       => $imagePath,
            'colors'                      => $colors,
            'description'                 => $request->description,
        ]);

        return redirect()
            ->route('admin.products')
            ->with('success', 'Product added successfully');
    }



    /* =========================
        EDIT PRODUCT
    ========================= */
    public function edit(Product $product)
    {
        $categories    = Category::where('status', 'active')->get();
        $subcategories = SubCategory::where('status', 'active')->get();
        $brands        = Brand::where('status', 'active')->get();

        return view('admin.products.form', compact(
            'product',
            'categories',
            'subcategories',
            'brands'
        ));
    }

    /* =========================
        UPDATE PRODUCT
    ========================= */
    public function update(Request $request, Product $product)
    {
        $request->validate([
            'category_id'       => 'nullable|exists:categories,id',
            'subcategory_id'    => 'nullable|exists:subcategories,id',
            'brand_id'          => 'nullable|exists:brands,id',
            'quantity_unit_id'  => 'nullable|exists:quantity_units,id',
            'store_price'       => 'nullable|numeric|min:0',
            'customer_price'    => 'nullable|numeric|min:0',
            'store_discount_type'  => 'nullable|in:percentage,amount',
            'store_discount_value' => 'nullable|numeric|min:0',
            'customer_discount_type'  => 'nullable|in:percentage,amount',
            'customer_discount_value' => 'nullable|numeric|min:0',
            'status'            => 'required|in:active,inactive',
            'image'             => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
            'description'       => 'nullable|string|max:2000',
            'colors'            => 'nullable|array',
            'colors.*.name'     => 'required_with:colors|string|max:50',
            'colors.*.hex'      => ['required_with:colors','regex:/^#[A-Fa-f0-9]{6}$/'],
        ]);

        // Check for duplicates, excluding current product
        $exists = Product::where('category_id', $request->category_id)
            ->when($request->subcategory_id, function ($q) use ($request) {
                $q->where('subcategory_id', $request->subcategory_id);
            }, function ($q) {
                $q->whereNull('subcategory_id');
            })
            ->where('brand_id', $request->brand_id)
            ->where('quantity_unit_id', $request->quantity_unit_id)
            ->where('id', '!=', $product->id) // Exclude current product
            ->first();

        if ($exists) {
            return back()
                ->withInput()
                ->withErrors([
                    'product' => 'Product already exists with the same category, subcategory, brand, and quantity unit.'
                ]);
        }

        // Handle image upload
        $imagePath = $product->image; // Keep existing image by default
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = 'product_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'uploads/products';

            if (!file_exists(public_path($uploadPath))) {
                mkdir(public_path($uploadPath), 0755, true);
            }

            $file->move(public_path($uploadPath), $filename);
            $imagePath = $uploadPath . '/' . $filename;
        }

        $colors = $request->colors ? array_values($request->colors) : null;

        $quantityUnitName = $request->quantity_unit_id
                            ? optional(QuantityUnit::find($request->quantity_unit_id))->name
                            : null;

        // Update the product
        $product->update([
            'category_id'                => $request->category_id,
            'subcategory_id'             => $request->subcategory_id,
            'brand_id'                   => $request->brand_id,
            'category'                   => optional(Category::find($request->category_id))->name,
            'subcategory'                => optional(SubCategory::find($request->subcategory_id))->name,
            'brand'                      => optional(Brand::find($request->brand_id))->name,
            'quantity_unit_id'           => $request->quantity_unit_id,
            'quantity_unit'              => $quantityUnitName,
            'store_price'                => $request->store_price,
            'store_discount_type'         => $request->store_discount_type,
            'store_discount_value'        => $request->store_discount_value,
            'store_price_assigned_by'    => auth()->id(),
            'store_price_assigned_at'    => now(),
            'customer_price'             => $request->customer_price,
            'customer_discount_type'      => $request->customer_discount_type,
            'customer_discount_value'     => $request->customer_discount_value,
            'customer_price_assigned_by' => auth()->id(),
            'customer_price_assigned_at' => now(),
            'status'                     => $request->status,
            'image'                      => $imagePath,
            'colors'                     => $colors,
            'description'                => $request->description,
        ]);

        return redirect()
            ->route('admin.products')
            ->with('success', 'Product updated successfully');
    }



    /* =========================
        DELETE PRODUCT
    ========================= */
    public function destroy(Product $product)
    {
        $product->delete();

        return redirect()
            ->route('admin.products')
            ->with('success', 'Product deleted successfully');
    }

    public function getSubcategories(Category $category)
    {
        return response()->json(
            $category->subcategories()
                     ->where('status','active')
                     ->select('id','name')
                     ->get()
        );
    }

    public function getByCategory($categoryId)
    {
        $units = QuantityUnit::whereJsonContains('category_ids', $categoryId)
                            ->where('status', 'active')
                            ->get(['id', 'name']);
        return response()->json($units);
    }

    
    public function getBrandsByCategory($categoryId)
    {
        return Brand::where(function ($q) use ($categoryId) {
                $q->whereJsonContains('category_ids', (int)$categoryId)
                ->orWhereJsonContains('category_ids', (string)$categoryId);
            })
            ->where('status', 'active')
            ->select('id', 'name')
            ->get();
    }



    public function updatePricing(Request $request, Product $product)
    {
        $request->validate([
            'store_price' => 'required|numeric|min:0',
            'customer_price' => 'required|numeric|min:0',

            'store_discount_type' => 'nullable|in:percentage,amount',
            'store_discount_value' => 'nullable|numeric|min:0',

            'customer_discount_type' => 'nullable|in:percentage,amount',
            'customer_discount_value' => 'nullable|numeric|min:0',

            'status' => 'required|in:active,inactive',
        ]);

        $product->update([
            'store_price' => $request->store_price,
            'store_discount_type' => $request->store_discount_type,
            'store_discount_value' => $request->store_discount_value ?? 0,
            'store_price_assigned_by' => auth()->id(),
            'store_price_assigned_at' => now(),

            'customer_price' => $request->customer_price,
            'customer_discount_type' => $request->customer_discount_type,
            'customer_discount_value' => $request->customer_discount_value ?? 0,
            'customer_price_assigned_by' => auth()->id(),
            'customer_price_assigned_at' => now(),

            'status' => $request->status,
        ]);

        return back()->with('success', 'Product pricing & discounts updated successfully');
    }



    /* =========================
       Categories
    ========================= */
    public function CategoryIndex()
    {
        $categories = Category::latest()->get();
        return view('admin.products.categories', compact('categories'));
    }

    public function CategoryStore(Request $request)
    {
        $request->validate([
            'name'  => 'required|string|max:255|unique:categories,name',
            'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        $imagePath = null;

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = 'category_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'uploads/categories';
            $file->move(public_path($uploadPath), $filename);
            $imagePath = $uploadPath . '/' . $filename;
        }

        Category::create([
            'name'   => $request->name,
            'image'  => $imagePath,
            'status' => 'active',
        ]);

        return redirect()->back()->with('success', 'Category added successfully');
    }

    public function CategoryUpdate(Request $request, Category $category)
    {
        $request->validate([
            'name'   => 'required|string|max:255|unique:categories,name,' . $category->id,
            'status' => 'required|in:active,inactive',
            'image'  => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        if ($request->hasFile('image')) {

            if ($category->image && file_exists(public_path($category->image))) {
                unlink(public_path($category->image));
            }
            $file = $request->file('image');
            $filename = 'category_' . $category->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'uploads/categories';
            $file->move(public_path($uploadPath), $filename);
            $category->image = $uploadPath . '/' . $filename;
        }

        $category->update([
            'name'   => $request->name,
            'status' => $request->status,
        ]);

        return redirect()->back()->with('success', 'Category updated successfully');
    }

    public function CategoryDestroy(Category $category)
    {
        $category->delete();
        return redirect()->back()->with('success', 'Category deleted successfully');
    }

    /* =========================
    Sub Categories
    ========================= */
    public function SubCategoryIndex()
    {
        $categories = Category::where('status', 'active')->get();
        $subcategories = SubCategory::with('category')->latest()->get();

        return view('admin.products.subcategories',
            compact('categories', 'subcategories'));
    }

    public function SubCategoryStore(Request $request)
    {
        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'name'        => 'required|string|max:255|unique:subcategories,name',
            'image'       => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        $imagePath = null;

        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $filename = 'subcategory_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'uploads/subcategories';

            $file->move(public_path($uploadPath), $filename);
            $imagePath = $uploadPath . '/' . $filename;
        }

        SubCategory::create([
            'category_id' => $request->category_id,
            'name'        => $request->name,
            'image'       => $imagePath,
            'status'      => 'active',
        ]);

        return redirect()->back()->with('success', 'Sub-Category added successfully');
    }


    public function SubCategoryUpdate(Request $request, SubCategory $subcategory)
    {
        $request->validate([
            'category_id' => 'required|exists:categories,id',
            'name'        => 'required|string|max:255|unique:subcategories,name,' . $subcategory->id,
            'status'      => 'required|in:active,inactive',
            'image'       => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        if ($request->hasFile('image')) {

            if ($subcategory->image && file_exists(public_path($subcategory->image))) {
                unlink(public_path($subcategory->image));
            }

            $file = $request->file('image');
            $filename = 'subcategory_' . $subcategory->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'uploads/subcategories';

            $file->move(public_path($uploadPath), $filename);
            $subcategory->image = $uploadPath . '/' . $filename;
        }

        $subcategory->update([
            'category_id' => $request->category_id,
            'name'        => $request->name,
            'status'      => $request->status,
        ]);

        return redirect()->back()->with('success', 'Sub-Category updated successfully');
    }


    public function SubCategoryDestroy(Subcategory $subcategory)
    {
        $subcategory->delete();
        return redirect()->back()->with('success', 'Sub-Category deleted successfully');
    }

    /* =========================
        Quantity Units
    ========================= */

    public function QuantityUnitIndex()
    {
        $categories = Category::where('status', 'active')->get();
        $quantityUnits = QuantityUnit::latest()->get(); // no need for 'with' now
        return view('admin.products.quantity_units', compact('categories', 'quantityUnits'));
    }

    public function QuantityUnitStore(Request $request)
    {
        $request->validate([
            'category_ids' => 'required|array',
            'category_ids.*' => 'exists:categories,id',
            'name' => 'required|string|max:100|unique:quantity_units,name',
            'status' => 'required|in:active,inactive'
        ]);

        QuantityUnit::create([
            'name' => $request->name,
            'category_ids' => $request->category_ids,
            'status' => $request->status
        ]);

        return redirect()->back()->with('success', 'Quantity unit added successfully');
    }

    public function QuantityUnitUpdate(Request $request, QuantityUnit $quantityUnit)
    {
        $request->validate([
            'category_ids' => 'required|array',
            'category_ids.*' => 'exists:categories,id',
            'name' => 'required|string|max:100|unique:quantity_units,name,' . $quantityUnit->id,
            'status' => 'required|in:active,inactive'
        ]);

        $quantityUnit->update([
            'name' => $request->name,
            'category_ids' => $request->category_ids,
            'status' => $request->status
        ]);

        return redirect()->back()->with('success', 'Quantity unit updated successfully');
    }

    public function QuantityUnitDestroy(QuantityUnit $quantityUnit)
    {
        $quantityUnit->delete();
        return redirect()->back()->with('success', 'Quantity unit deleted successfully');
    }


    /* =========================
        Brands
    ========================= */

    public function BrandIndex()
    {
        $categories = Category::where('status', 'active')->get();
        $brands = Brand::latest()->get();
        return view('admin.products.brand', compact('categories', 'brands'));
    }

    public function BrandStore(Request $request)
    {
        $request->validate([
            'category_ids'   => 'required|array',
            'category_ids.*' => 'exists:categories,id',
            'name'           => 'required|string|max:100|unique:brands,name',
            'status'         => 'required|in:active,inactive'
        ]);

        Brand::create([
            'name'         => $request->name,
            'category_ids' => $request->category_ids,
            'status'       => $request->status
        ]);

        return redirect()->back()->with('success', 'Brand added successfully');
    }

    public function BrandUpdate(Request $request, Brand $brand)
    {
        $request->validate([
            'category_ids'   => 'required|array',
            'category_ids.*' => 'exists:categories,id',
            'name'           => 'required|string|max:100|unique:brands,name,' . $brand->id,
            'status'         => 'required|in:active,inactive'
        ]);

        $brand->update([
            'name'         => $request->name,
            'category_ids' => $request->category_ids,
            'status'       => $request->status
        ]);

        return redirect()->back()->with('success', 'Brand updated successfully');
    }

    public function BrandDestroy(Brand $brand)
    {
        $brand->delete();
        return redirect()->back()->with('success', 'Brand deleted successfully');
    }

    public function reviewIndex()
    {
        $reviews = ProductReview::with(['product','order','customer','store'])
            ->latest()
            ->paginate(15);

        return view('admin.product-reviews.index', compact('reviews'));
    }

    public function reviewStatus($id)
    {
        $review = ProductReview::findOrFail($id);

        $review->status = $review->status === 'active' ? 'inactive' : 'active';
        $review->save();

        return redirect()->back()->with('success', 'Review status updated successfully');
    }


}
