<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Address;
use App\Models\Customer;
use App\Models\Product;
use App\Models\Rider;
use App\Models\StoreProduct;
use Illuminate\Http\Request;
use App\Models\Warehouse;
use App\Models\Store;
use App\Models\WarehouseProduct;
use Illuminate\Support\Facades\DB;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\Brand;
use App\Models\DeliveryPrice;

class UserController extends Controller
{
    /* ===============================
       COMMON LISTING
    =============================== */

    public function warehouses(Request $request)
    {
        $status = $request->get('status');
        $query = Warehouse::with([
            'state',
            'district',
            'bank',
            'products.category',
            'products.subcategory',
            'products.brand',
            'products.quantityUnit'
        ])->orderBy('id', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        return view('admin.users.index', [
            'type'   => 'warehouse',
            'items'  => $query->get(),
            'status' => $status
        ]);
    }

    public function stores(Request $request)
    {
        $status = $request->get('status');

        $query = Store::with([
            'state',
            'district',
            'bank',
            'products.category',
            'products.subcategory',
            'products.brand',
            'products.quantityUnit'
        ])->orderBy('id', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        return view('admin.users.index', [
            'type'   => 'store',
            'items'  => $query->get(),
            'status' => $status
        ]);
    }

    public function riders(Request $request)
    {
        $status = $request->get('status');

        $query = Rider::with(['state', 'district', 'bank'])
            ->orderBy('id', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        $riders = $query->get();

        return view('admin.riders.index', compact('riders', 'status'));
    }


    /* ===============================
       HELPER METHOD
    =============================== */

    private function resolveModel($type)
    {
        return match ($type) {
            'warehouse' => Warehouse::class,
            'store'     => Store::class,
            'rider'     => Rider::class,
            default     => abort(404, 'Invalid user type'),
        };
    }

    /* ===============================
       COMMON DETAILS PAGE
    =============================== */

    public function show($type, $id)
    {
        $item = $this->resolveModel($type)::with('products')->findOrFail($id);

        return view('admin.users.show', compact('type', 'item'));
    }

    public function riderShow($id)
    {
        $rider = Rider::with([
            'state',
            'district',
            'bank'
        ])->findOrFail($id);

        $type = 'rider';

        return view('admin.riders.show', compact('rider','type'));
    }

    /* ===============================
       APPROVE / REJECT USER
    =============================== */

    public function approve(Request $request, $type, $id)
    {
        $request->validate([
            'admin_remark' => 'nullable|string|max:500',
        ]);

        $user = $this->resolveModel($type)::findOrFail($id);

        $user->update([
            'status'        => 'active',
            'admin_remark'  => $request->admin_remark,
            'approved_at'   => now(),
            'approved_by'   => auth()->id(),
        ]);

        return back()->with('success', ucfirst($type).' approved successfully');
    }

    public function reject(Request $request, $type, $id)
    {
        $request->validate([
            'admin_remark' => 'required|string|max:500',
        ]);

        $user = $this->resolveModel($type)::findOrFail($id);

        $user->update([
            'status'        => 'rejected',
            'admin_remark'  => $request->admin_remark,
            'approved_at'   => now(),
            'approved_by'   => auth()->id(),
        ]);

        return back()->with('success', ucfirst($type).' rejected successfully');
    }

    /* ===============================
       PRODUCT APPROVAL
    =============================== */

    private function resolveProductModel($type)
    {
        return match ($type) {
            'warehouse' => WarehouseProduct::class,
            'store'     => StoreProduct::class,
            default     => abort(404, 'Invalid type'),
        };
    }

    public function approveProduct(Request $request, $type, $id)
    {
        $productModel = $this->resolveProductModel($type);
        $item = $productModel::findOrFail($id);

        $product = Product::where('category_id', $item->category_id)
            ->where('subcategory_id', $item->subcategory_id)
            ->where('brand_id', $item->brand_id)
            ->where('quantity_unit_id', $item->quantity_unit_id)
            ->first();

        $categoryName = optional(Category::find($item->category_id))->name;
        $subcategoryName = optional(SubCategory::find($item->subcategory_id))->name;
        $brandName = optional(Brand::find($item->brand_id))->name;
        $quantityUnitName = optional($item->quantityUnit)->name ?? $item->quantity_unit;

        $validationRules = [];
        if (!$product || !$product->image) {
            $validationRules['image'] = 'required|image|mimes:jpg,jpeg,png,webp|max:2048';
        }
        if (!$product || !$product->store_price) {
            $validationRules['store_price'] = 'required|numeric|min:0';
        }
        if (!$product || !$product->customer_price) {
            $validationRules['customer_price'] = 'required|numeric|min:0';
        }
        $request->validate($validationRules);

        $validator = \Validator::make($request->all(), $validationRules);

        DB::transaction(function() use ($request, $item, $product, $categoryName, $subcategoryName, $brandName, $quantityUnitName, $type) {

            $imagePath = $product->image ?? null;
            if ($request->hasFile('image')) {
                if ($product && $product->image && file_exists(public_path($product->image))) {
                    unlink(public_path($product->image));
                }

                $file = $request->file('image');
                $filename = 'product_' . time() . '.' . $file->getClientOriginalExtension();
                $uploadPath = 'uploads/products';
                if (!file_exists(public_path($uploadPath))) {
                    mkdir(public_path($uploadPath), 0755, true);
                }
                $file->move(public_path($uploadPath), $filename);
                $imagePath = $uploadPath . '/' . $filename;
            }

            $productData = [
                'category_id' => $item->category_id,
                'subcategory_id' => $item->subcategory_id,
                'brand_id' => $item->brand_id,
                'quantity_unit_id' => $item->quantity_unit_id,
                'category' => $categoryName,
                'subcategory' => $subcategoryName,
                'brand' => $brandName,
                'quantity_unit' => $quantityUnitName,
                'image' => $imagePath,
                'store_price' => $request->store_price ?? ($product->store_price ?? 0),
                'store_price_assigned_by' => auth('admin')->id(),
                'store_price_assigned_at' => now(),
                'customer_price' => $request->customer_price ?? ($product->customer_price ?? 0),
                'customer_price_assigned_by' => auth('admin')->id(),
                'customer_price_assigned_at' => now(),
                'status' => 'active',
            ];

            if (!$product) {
                $product = Product::create($productData);
            } else {
                $product->update($productData);
            }

            $item->update([
                'product_id' => $product->id,
                'status' => 'approved',
                'admin_remark' => $request->admin_remark ?? null,
            ]);

            $productLabel = trim(
                ($subcategoryName ?? 'Product') . ' by ' .
                ($brandName ?? 'Brand') . ' (' .
                ($quantityUnitName ?? 'Unit') . ')'
            );

            if ($type === 'warehouse') {
                $userId = $item->warehouse_id;
            } elseif ($type === 'store') {
                $userId = $item->store_id;
            } else {
                $userId = null;
            }

            if ($userId) {
                $route   = "{$type}/category/products";
                $routeId = $item->category_id;

                $this->sendNotification(
                    'Product Approved',
                    "{$productLabel} is now approved and live.",
                    $type,
                    $userId,
                    $route,
                    $routeId
                );

            }
        });

        return back()->with('success', ucfirst($type).' product approved successfully!');
    }

    public function rejectProduct(Request $request, $type, $id)
    {
        $request->validate([
            'admin_remark' => 'nullable|string|max:500',
        ]);

        if (!in_array($type, ['warehouse', 'store'])) {
            abort(404, 'Invalid product type');
        }

        $model = $type === 'store' ? StoreProduct::class : WarehouseProduct::class;

        $item = $model::findOrFail($id);

        $item->update([
            'status' => 'rejected',
            'admin_remark' => $request->admin_remark,
        ]);

        $remarkText = $request->admin_remark 
            ? " Review: \"{$request->admin_remark}\"" 
            : '';

        $userId = $type === 'warehouse' ? $item->warehouse_id : $item->store_id;

        if ($userId) {

            $route   = "{$type}/product/edit";   // route path
            $routeId = $item->id;                // product id

            $this->sendNotification(
                'Product Rejected',
                "Product has been rejected by admin. Please change it and re-quote.{$remarkText}",
                $type,
                $userId,
                $route,
                $routeId
            );
        }
        return back()->with('success', ucfirst($type).' product rejected successfully');
    }

    private function sendNotification($title, $content, $userType, $userId, $route = null, $routeId = null)
    {
        $linkData = null;

        if ($route) {
            $linkData = $routeId !== null
                ? json_encode(['route' => $route, 'route_id' => $routeId], JSON_UNESCAPED_SLASHES)
                : json_encode(['route' => $route], JSON_UNESCAPED_SLASHES);
        }


        DB::table('notifications')->insert([
            'title'      => $title,
            'content'    => $content,
            'user_type'  => $userType,
            'user_id'    => $userId,
            'status'     => 'active',
            'link'       => $linkData,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        /* ========== 2. Get active FCM tokens ========== */
        $tokens = DB::table('fcm_tokens')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('notify_status', 'active')
            ->pluck('fcm_token')
            ->toArray();

        if (empty($tokens)) {
            return; 
        }

        /* ========== 3. Firebase HTTP v1 Auth ========== */
        $credentialsPath = base_path(env('FIREBASE_CREDENTIALS'));
        $projectId       = env('FIREBASE_PROJECT_ID');

        $client = new \Google_Client();
        $client->setAuthConfig($credentialsPath);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');

        $accessToken = $client->fetchAccessTokenWithAssertion();

        if (!isset($accessToken['access_token'])) {
            \Log::error('Firebase token error', $accessToken);
            return;
        }

        $bearerToken = $accessToken['access_token'];

        $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

        foreach ($tokens as $fcmToken) {

            $dataPayload = [
                "route"     => (string) $route,
                "type"      => "notification",
                "user_type" => (string) $userType,
                "status"    => "active",
            ];

            if ($routeId !== null) {
                $dataPayload["route_id"] = (string) $routeId;
            }


            $payload = [
                "message" => [
                    "token" => $fcmToken,
                    "notification" => [
                        "title" => $title,
                        "body"  => $content,
                    ],
                    "data" => $dataPayload,
                    "android" => [
                        "priority" => "high",
                        "notification" => [
                            "sound" => "default",
                            "click_action" => "FLUTTER_NOTIFICATION_CLICK"
                        ]
                    ]
                ]
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Authorization: Bearer {$bearerToken}",
                "Content-Type: application/json"
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

            $response = curl_exec($ch);

            if ($response === false) {
                \Log::error('Firebase Push Error: ' . curl_error($ch));
            }

            curl_close($ch);
        }
    }

    public function approveBank(Request $request, $type, $id)
    {
        $user = $this->resolveModel($type)::findOrFail($id);

        if ($user->bank_status !== 'pending' || empty($user->pending_bank_data)) {
            return back()->with('error', 'No pending bank request to approve.');
        }

        $pending = $user->pending_bank_data;

        // Move pending data to main bank fields
        $user->update([
            'bank_id'             => $pending['bank_id'],
            'bank_holder_name'    => $pending['bank_holder_name'],
            'bank_account_number' => $pending['bank_account_number'],
            'ifsc_code'           => $pending['ifsc_code'],
            'micr_no'             => $pending['micr_no'] ?? null,
            'upi_id'              => $pending['upi_id'] ?? null,
            'passbook_file'       => $pending['passbook_file'],
            'bank_status'         => 'approved',
            'bank_approved_at'    => now(),
            'bank_approved_by'    => auth()->id(),
            'edit_bank'           => 0,
            'bank_edit_reason'    => null,
            'pending_bank_data'   => null,
            'admin_remark'        => $request->remark ?? null
        ]);

        $route = "{$type}/bank-details";

        $this->sendNotification(
            'Bank Details Update Approved',
            'Your request to update bank account details has been approved by admin.',
            $type,
            $user->id,
            $route,
            null
        );

        return back()->with('success', ucfirst($type).' bank details approved successfully.');
    }

    public function rejectBank(Request $request, $type, $id)
    {
        $user = $this->resolveModel($type)::findOrFail($id);

        if ($user->bank_status !== 'pending' || empty($user->pending_bank_data)) {
            return back()->with('error', 'No pending bank request to reject.');
        }

        if (!empty($user->pending_bank_data['passbook_file']) &&
            file_exists(public_path('storage/' . $user->pending_bank_data['passbook_file']))) {
            unlink(public_path('storage/' . $user->pending_bank_data['passbook_file']));
        }

        // Reject request, keep old bank details intact
        $user->update([
            'bank_status'        => 'rejected',
            'bank_approved_at'   => now(),
            'bank_approved_by'   => auth()->id(),
            // 'edit_bank'          => 1, // allow user to edit again
            'pending_bank_data'  => null,
            'bank_edit_reason'   => null,
            'admin_remark'       => $request->remark ?? null
        ]);

        $route = "{$type}/bank-details";

        $this->sendNotification(
            'Bank Details Update Rejected',
            'Your request to update bank account details has been rejected by admin.',
            $type,
            $user->id,
            $route,
            null
        );

        return back()->with('success', ucfirst($type).' bank details rejected successfully.');
    }

    public function approveRider(Request $request, $id)
    {
        $request->validate([
            'admin_remark' => 'nullable|string|max:500',
        ]);

        $rider = Rider::findOrFail($id);

        if ($rider->status !== 'pending_admin') {
            return back()->with('error', 'Rider is not pending for approval.');
        }

        $riderCode = $this->generateRiderCode();

        $rider->update([
            'status'       => 'active',
            'rider_code'   => $riderCode,
            'admin_remark' => $request->admin_remark,
            'approved_at'  => now(),
            'approved_by'  => auth()->id(),
        ]);

        return redirect()->route('admin.riders.show', $id)->with('success', 'Rider approved successfully.');

    }
    private function generateRiderCode()
    {
        $year = date('Y');
        $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $length = 6;

        do {
            $randomString = '';
            for ($i = 0; $i < $length; $i++) {
                $randomString .= $characters[random_int(0, strlen($characters) - 1)];
            }
            $code = 'BR' . $year . $randomString;
        } while (Rider::where('rider_code', $code)->exists());

        return $code;
    }
    public function rejectRider(Request $request, $id)
    {
        $request->validate([
            'admin_remark' => 'required|string|max:500',
        ]);

        $rider = Rider::findOrFail($id);

        if ($rider->status !== 'pending_admin') {
            return back()->with('error', 'Rider is not pending for approval.');
        }

        $rider->update([
            'status'       => 'rejected',
            'admin_remark' => $request->admin_remark,
            'approved_at'  => now(),
            'approved_by'  => auth()->id(),
        ]);

        return redirect()->route('admin.riders.show', $id)->with('success', 'Rider rejected successfully.');

    }

    public function approveRate($id)
    {
        $rider = Rider::findOrFail($id);

        $rider->update([
            'rate_change_status' => 'approved',
            'rate_approved_at'   => now(),
            'rate_approved_by'   => auth()->id(),
        ]);

        $route = "rider/price-quote";

        $this->sendNotification(
            'Rate Change Approved',
            'Your requested rate change has been approved by admin.',
            'rider',
            $rider->id,
            $route,   // route
            null      // no route_id
        );

        return back()->with('success', 'Rate approved successfully');
    }

    public function rejectRate(Request $request, $id)
    {
        $rider = Rider::findOrFail($id);

        $rider->update([
            'rate_change_status' => 'rejected',
            'rate_approved_at'   => now(),
            'rate_approved_by'   => auth()->id(),
        ]);

        $route = "rider/price-quote";

        $this->sendNotification(
            'Rate Change Rejected',
            'Your requested rate change has been rejected by admin.',
            'rider',
            $rider->id,
            $route,   // route
            null      // no route_id
        );

        return back()->with('error', 'Rate rejected');
    }

    public function customers(Request $request)
    {
        $status = $request->get('status'); // get the status filter

        $query = DB::table('customers')
            ->select('id', 'name', 'email', 'mobile_number', 'gender', 'status', 'profile_image')
            ->orderBy('created_at', 'desc');

        if ($status) {
            $query->where('status', $status);
        }

        $customers = $query->get();

        return view('admin.customers.index', compact('customers', 'status'));
    }

    public function customerDetails($id)
    {
        $customer = Customer::findOrFail($id);

        $addresses = Address::with(['district', 'state'])
            ->where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->orderByDesc('is_default')
            ->get();

        return view('admin.customers.show', compact('customer', 'addresses'));
    }

    public function blockCustomer($id)
    {
        Customer::where('id', $id)->update([
            'status' => 'blocked'
        ]);

        return back()->with('success', 'Customer blocked successfully');
    }

    public function unblockCustomer($id)
    {
        Customer::where('id', $id)->update([
            'status' => 'active'
        ]);

        return back()->with('success', 'Customer unblocked successfully');
    }

    /**
     * 📌 Index page (Blade)
     */
    public function deliveryIndex()
    {
        $prices = DeliveryPrice::orderBy('is_default', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('admin.delivery-prices.index', compact('prices'));
    }

    public function deliveryStore(Request $request)
    {
        $request->validate([
            'store_delivery_price' => 'required|numeric|min:0',
            'customer_delivery_price' => 'required|numeric|min:0',
            'is_default' => 'nullable',
        ]);

        DB::transaction(function () use ($request) {

            // Only one default allowed
            if ($request->has('is_default')) {
                DeliveryPrice::where('is_default', true)
                    ->update(['is_default' => false]);
            }

            DeliveryPrice::create([
                'store_delivery_price' => $request->store_delivery_price,
                'customer_delivery_price' => $request->customer_delivery_price,
                'is_default' => $request->has('is_default'),
                'created_by' => auth()->id(),
                'created_by_type' => 'admin',
            ]);
        });

        return redirect()
            ->back()
            ->with('success', 'Delivery price added successfully');
    }

    public function deliveryUpdate(Request $request, $id)
    {
        $price = DeliveryPrice::findOrFail($id);

        $request->validate([
            'store_delivery_price' => 'required|numeric|min:0',
            'customer_delivery_price' => 'required|numeric|min:0',
            'is_default' => 'nullable',
        ]);

        DB::transaction(function () use ($request, $price) {

            if ($request->has('is_default')) {
                DeliveryPrice::where('is_default', true)
                    ->where('id', '!=', $price->id)
                    ->update(['is_default' => false]);
            }

            $price->update([
                'store_delivery_price' => $request->store_delivery_price,
                'customer_delivery_price' => $request->customer_delivery_price,
                'is_default' => $request->has('is_default'),
            ]);
        });

        return redirect()
            ->back()
            ->with('success', 'Delivery price updated successfully');
    }

    public function deliveryDestroy($id)
    {
        $deliveryPrice = DeliveryPrice::find($id);

        if (!$deliveryPrice) {
            return redirect()->back()->with('error', 'Delivery price not found.');
        }

        if ($deliveryPrice->is_default == 1) {
            return redirect()->back()->with(
                'error',
                'Default delivery price cannot be deleted.'
            );
        }

        $deliveryPrice->delete();
        return redirect()->back()->with(
            'success',
            'Delivery price deleted successfully.'
        );
    }

}

