<?php

namespace App\Http\Controllers;

use App\Models\Address;
use App\Models\State;
use App\Models\District;
use App\Services\GeoService;
use App\Helpers\GeoHelper;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Validation\Rule;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Mail;
use App\Mail\CustomerOtpMail;
use Illuminate\Support\Facades\Http;


class CustomerController extends Controller
{
    // ================= REGISTER =================
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string',
            'email' => 'required|email|unique:customers,email',
            'mobile_number' => 'required|unique:customers,mobile_number',
            'password' => 'required|min:8',
            'door_no' => 'required',
            'street_name' => 'required',
            'area' => 'required',
            'city' => 'required',
            'district' => 'required',
            'state' => 'required',
            'pincode' => 'required',
            'country' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $otp = '1111'; // fixed for testing
        $registrationToken = Str::uuid()->toString();

        Cache::put(
            'customer_register_' . $registrationToken,
            [
                'otp' => $otp,
                'otp_created_at' => now(),
                'data' => $request->all()
            ],
            now()->addMinutes(10)
        );

        return response()->json([
            'success' => true,
            'message' => 'OTP generated. Please verify.',
            'registration_token' => $registrationToken,
            'otp' => $otp // TEMP (testing only)
        ]);
    }

    // ================= OTP VERIFY =================
    // public function verifyOtp(Request $request)
    // {
    //     $request->validate([
    //         'registration_token' => 'required',
    //         'otp' => 'required|digits:4'
    //     ]);

    //     $cacheKey = 'customer_register_' . $request->registration_token;
    //     $cached = Cache::get($cacheKey);

    //     if (!$cached) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Session expired. Please register again.'
    //         ], 400);
    //     }

    //     if (
    //         $cached['otp'] !== $request->otp ||
    //         now()->diffInMinutes($cached['otp_created_at']) > 5
    //     ) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Invalid or expired OTP'
    //         ], 400);
    //     }

    //     $data = $cached['data'];

    //     $customer = Customer::create([
    //         'name' => $data['name'],
    //         'email' => $data['email'],
    //         'mobile_number' => $data['mobile_number'],
    //         'password' => Hash::make($data['password']),

    //         'door_no' => $data['door_no'],
    //         'street_name' => $data['street_name'],
    //         'area' => $data['area'],
    //         'city' => $data['city'],
    //         'district' => $data['district'],
    //         'state' => $data['state'],
    //         'pincode' => $data['pincode'],
    //         'country' => $data['country'],
    //         'status' => 'active'
    //     ]);

    //     Cache::forget($cacheKey);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'OTP verified. Customer registered successfully.',
    //         'customer_id' => $customer->id
    //     ]);
    // }


    // ================= LOGIN =================
    // public function login(Request $request)
    // {
    //     $request->validate([
    //         'login' => 'required',
    //         'password' => 'required|string|min:8'
    //     ]);

    //     $customer = Customer::where('email', $request->login)
    //         ->orWhere('mobile_number', $request->login)
    //         ->first();

    //     if (!$customer) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Account not found'
    //         ], 404);
    //     }

    //     if (!Hash::check($request->password, $customer->password)) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Invalid password'
    //         ], 401);
    //     }

    //     if ($customer->status !== 'active') {

    //         $message = match ($customer->status) {
    //             'pending_otp'   => 'OTP verification required',
    //             'pending_admin' => 'Account pending admin approval',
    //             'rejected'      => 'Account rejected by admin',
    //             default         => 'Account inactive',
    //         };

    //         return response()->json([
    //             'success' => false,
    //             'message' => $message
    //         ], 403);
    //     }

    //     if ($request->filled('fcm_token')) {
    //         DB::table('fcm_tokens')->updateOrInsert(
    //             [
    //                 'user_type' => 'customer',
    //                 'user_id'   => $customer->id,
    //             ],
    //             [
    //                 'fcm_token'     => $request->fcm_token,
    //                 'notify_status' => 'active',
    //                 'device_type'   => $request->device_type ?? 'android',
    //                 'updated_at'    => now(),
    //                 'created_at'    => now()
    //             ]
    //         );
    //     }


    //     $token = JWTAuth::customClaims([
    //         'exp' => now()->addDays(30)->timestamp
    //     ])->fromUser($customer);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Login successful',
    //         'token' => $token,
    //         'token_type' => 'bearer',
    //         'expires_in' => JWTAuth::factory()->getTTL() * 60,  
    //         'customer' => [
    //             'id' => $customer->id,
    //             'name' => $customer->name,
    //             'email' => $customer->email,
    //             'mobile_number' => $customer->mobile_number,
    //         ]
    //     ]);
    // }

    public function loginOrRegister(Request $request)
    {
        $request->validate([
            'login' => 'required',
            'name'  => 'sometimes|string|max:255'
        ]);

        $login = $request->login;
        $isEmail = filter_var($login, FILTER_VALIDATE_EMAIL);

        $customer = Customer::where('email', $login)
            ->orWhere('mobile_number', $login)
            ->first();

        if (!$customer) {
            if (!$isEmail) {
                if (!$request->filled('name')) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Name is required for new mobile users'
                    ], 422);
                }
            }

            $customer = Customer::create([
                'name' => $request->name ?? 'New User',
                'email' => $isEmail ? $login : null,
                'mobile_number' => !$isEmail ? $login : null,
                'status' => 'active'
            ]);
        }

        if ($customer->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Your account is not active'
            ], 403);
        }

        if ($isEmail) {
            $otp = rand(100000, 999999);
            Mail::to($customer->email)->send(new CustomerOtpMail($otp, $customer->name));
        } else {
            $otp = '111111';
        }

        $customer->update([
            'otp' => $otp,
            'otp_created_at' => now()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'OTP sent successfully',
            'customer_id' => $customer->id
        ]);
    }

    public function verifyOtp(Request $request)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'otp'         => 'required',
            'fcm_token'   => 'sometimes|string',
            'device_type' => 'sometimes|string'
        ]);

        $customer = Customer::find($request->customer_id);

        if ($customer->otp != $request->otp) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid OTP'
            ], 401);
        }

        if (Carbon::parse($customer->otp_created_at)->addMinutes(5)->lt(now())) {
            return response()->json([
                'success' => false,
                'message' => 'OTP expired'
            ], 401);
        }

        $customer->update([
            'otp' => null,
            'otp_created_at' => null
        ]);

        if ($request->filled('fcm_token')) {
            DB::table('fcm_tokens')->updateOrInsert(
                [
                    'user_type' => 'customer',
                    'user_id'   => $customer->id
                ],
                [
                    'fcm_token'     => $request->fcm_token,
                    'notify_status' => 'active',
                    'device_type'   => $request->device_type ?? 'android',
                    'updated_at'    => now(),
                    'created_at'    => now()
                ]
            );
        }

        $token = JWTAuth::customClaims([
            'exp' => now()->addDays(30)->timestamp
        ])->fromUser($customer);

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'token' => $token,
            'token_type' => 'bearer',
            'expires_in' => JWTAuth::factory()->getTTL() * 60,
            'customer' => [
                'id'            => $customer->id,
                'name'          => $customer->name,
                'email'         => $customer->email,
                'mobile_number' => $customer->mobile_number
            ]
        ]);
    }

    // ================= PROFILE =================
    public function profile()
    {
        return response()->json([
            'customer' => auth('customer')->user()
        ]);
    }

    // ================= LOGOUT =================
    public function logout()
    {
        JWTAuth::invalidate(JWTAuth::getToken());
        return response()->json(['message' => 'Logged out']);
    }

    // ================= REFRESH =================
    public function refresh()
    {
        return response()->json([
            'token' => JWTAuth::refresh(JWTAuth::getToken())
        ]);
    }

     // ================= FORGOT PASSWORD - SEND OTP =================
    public function forgotPassword(Request $request)
    {
        $request->validate([
            'mobile_number' => 'required|digits:10|exists:customers,mobile_number'
        ]);

        $customer = Customer::where('mobile_number', $request->mobile_number)->first();

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Mobile number not registered'
            ], 404);
        }

        $otp = '1111'; // TEST OTP

        $customer->update([
            'otp' => $otp,
            'otp_created_at' => now()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'OTP sent successfully (TEST OTP: 1111)'
        ]);
    }

    // ================= FORGOT PASSWORD - VERIFY OTP =================
    public function verifyForgotOtp(Request $request)
    {
        $request->validate([
            'mobile_number' => 'required|digits:10|exists:customers,mobile_number',
            'otp' => 'required'
        ]);

        $customer = Customer::where('mobile_number', $request->mobile_number)
            ->where('otp', $request->otp)
            ->first();

        if (
            !$customer ||
            now()->diffInMinutes($customer->otp_created_at) > 5
        ) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired OTP'
            ], 400);
        }

        return response()->json([
            'success' => true,
            'message' => 'OTP verified successfully'
        ]);
    }


    // ================= RESET PASSWORD =================
    public function resetPassword(Request $request)
    {
        $request->validate([
            'mobile_number' => 'required|digits:10',
            'password' => 'required|min:8|confirmed'
        ]);

        $customer = Customer::where('mobile_number', $request->mobile_number)->first();

        if (!$customer) {
            \Log::info('Customer not found for mobile', [$request->mobile_number]);

            return response()->json([
                'success' => false,
                'message' => 'Customer not found'
            ], 404);
        }

        $customer->update([
            'password' => Hash::make($request->password),
            'otp' => null,
            'otp_created_at' => null
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password reset successfully'
        ]);
    }

    public function account()
    {
        $customer = auth('customer')->user();

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthenticated'
            ], 401);
        }

        $imageUrl = $customer->profile_image && file_exists(public_path($customer->profile_image))
            ? url($customer->profile_image)
            : match($customer->gender) {
                'male' => url('images/male.png'),
                'female' => url('images/female.png'),
                default => url('images/default.png'),
            };

        return response()->json([
            'success' => true,
            'data' => [
                'name' => $customer->name,
                'mobile_number' => $customer->mobile_number,
                'profile_image' => $imageUrl,
            ]
        ]);
    }

    public function getProfile()
    {
        $customer = auth('customer')->user();

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthenticated'
            ], 401);
        }

        if ($customer->profile_image && file_exists(public_path($customer->profile_image))) {
            $imageUrl = asset($customer->profile_image);
        } else {
            switch ($customer->gender) {
                case 'male':
                    $imageUrl = asset('images/male.png');
                    break;
                case 'female':
                    $imageUrl = asset('images/female.png');
                    break;
                default:
                    $imageUrl = asset('images/default.png');
            }
        }

        return response()->json([
            'success' => true,
            'data' => [
                'name'          => $customer->name,
                'email'         => $customer->email,
                'mobile_number' => $customer->mobile_number,
                'gender'        => $customer->gender,
                'profile_image' => $imageUrl,
            ]
        ]);
    }

    public function updateProfile(Request $request)
    {
        $customer = auth('customer')->user();

        if (!$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthenticated'
            ], 401);
        }

        $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => ['sometimes', 'email', Rule::unique('customers')->ignore($customer->id),],
            'mobile_number' => ['sometimes', 'digits:10',Rule::unique('customers')->ignore($customer->id),],
            'gender' => 'sometimes|in:male,female,other',
            'profile_image' => 'sometimes|image|mimes:jpg,jpeg,png,webp|max:2048',
        ]);

        $data = [];

        if ($request->has('name')) {
            $data['name'] = $request->name;
        }

        if ($request->has('email')) {
            $data['email'] = $request->email;
        }

        if ($request->has('mobile_number')) {
            $data['mobile_number'] = $request->mobile_number;
        }

        if ($request->has('gender')) {
            $data['gender'] = $request->gender;
        }

        if ($request->hasFile('profile_image')) {

            if ($customer->profile_image && file_exists(public_path($customer->profile_image))) {
                unlink(public_path($customer->profile_image));
            }

            $file = $request->file('profile_image');
            $filename = 'customer_' . $customer->id . '_' . time() . '.' . $file->getClientOriginalExtension();

            $uploadPath = 'uploads/customer/profile';
            $file->move(public_path($uploadPath), $filename);

            $data['profile_image'] = $uploadPath . '/' . $filename;
        }

        if (!empty($data)) {
            $customer->update($data);
        }

        return response()->json([
            'success' => true,
            'message' => 'Profile updated successfully',
            'data' => [
                'name'          => $customer->name,
                'email'         => $customer->email,
                'mobile_number' => $customer->mobile_number,
                'gender'        => $customer->gender,
                'profile_image' => $customer->profile_image
                    ? asset($customer->profile_image)
                    : null,
            ]
        ]);
    }
    
    // public function addAddress(Request $request)
    // {
    //     $customer = auth('customer')->user();

    //     $request->validate([
    //         'name' => 'required|string|max:255',
    //         'mobile_number' => 'required|digits:10',
    //         'door_no' => 'nullable|string|max:255',
    //         'street_name' => 'nullable|string|max:255',
    //         'area' => 'nullable|string|max:255',
    //         'city' => 'nullable|string|max:255',
    //         'district_id' => 'nullable|integer',
    //         'state_id' => 'nullable|integer',
    //         'pincode' => 'nullable|string|max:20',
    //         'latitude' => 'nullable|numeric',
    //         'longitude' => 'nullable|numeric',
    //         'is_default' => 'sometimes|boolean',
    //     ]);

    //     $hasAddress = Address::where('user_type', 'customer')
    //         ->where('user_id', $customer->id)
    //         ->exists();

    //     $isDefault = !$hasAddress ? 1 : ($request->is_default ?? 0);

    //     if ($isDefault) {
    //         Address::where('user_type', 'customer')
    //             ->where('user_id', $customer->id)
    //             ->update(['is_default' => 0]);
    //     }

    //     $address = Address::create([
    //         'user_type' => 'customer',
    //         'user_id' => $customer->id,
    //         'name' => $request->name,
    //         'mobile_number' => $request->mobile_number,
    //         'door_no' => $request->door_no,
    //         'street_name' => $request->street_name,
    //         'area' => $request->area,
    //         'city' => $request->city,
    //         'district_id' => $request->district_id,
    //         'state_id' => $request->state_id,
    //         'pincode' => $request->pincode,
    //         'latitude' => $request->latitude,
    //         'longitude' => $request->longitude,
    //         'is_default' => $isDefault,
    //     ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Address added successfully',
    //         'data' => $address
    //     ]);
    // }


    public function addAddress(Request $request)
    {       
        $customer = auth('customer')->user();

        $request->validate([
            'name' => 'required|string|max:255',
            'mobile_number' => 'required|digits:10',
            'door_no' => 'nullable|string|max:255',
            'street_name' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:255',
            'district_id' => 'nullable|integer',
            'state_id' => 'nullable|integer',
            'pincode' => 'nullable|string|max:20',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
            'is_default' => 'sometimes|boolean',
        ]);

        $hasAddress = Address::where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->exists();

        $isDefault = !$hasAddress ? 1 : ($request->is_default ?? 0);

        if ($isDefault) {
            Address::where('user_type', 'customer')
                ->where('user_id', $customer->id)
                ->update(['is_default' => 0]);
        }

        // ---------------- LAT / LON LOGIC START ----------------

        $latitude  = $request->latitude;
        $longitude = $request->longitude;

        // If frontend didn't send lat/lon → get from address
        if (!$latitude || !$longitude) {

            $stateName = State::where('id', $request->state_id)->value('name');
            $districtName = District::where('id', $request->district_id)->value('name');

            $fullAddress = implode(', ', array_filter([
                $request->door_no,
                $request->street_name,
                $request->city,
                $districtName,
                $stateName,
                $request->pincode,
                'India'
            ]));

            $geo = GeoService::getLatLongFromAddress($fullAddress);

            $latitude  = $geo['latitude'];
            $longitude = $geo['longitude'];
        }

        // ---------------- LAT / LON LOGIC END ----------------

        $address = Address::create([
            'user_type' => 'customer',
            'user_id' => $customer->id,
            'name' => $request->name,
            'mobile_number' => $request->mobile_number,
            'door_no' => $request->door_no,
            'street_name' => $request->street_name,
            'city' => $request->city,
            'district_id' => $request->district_id,
            'state_id' => $request->state_id,
            'pincode' => $request->pincode,
            'latitude' => $latitude,
            'longitude' => $longitude,
            'is_default' => $isDefault,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Address added successfully',
            'data' => $address
        ]);
    }

    public function listAddresses()
    {
        $customer = auth('customer')->user();

        $addresses = Address::where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->orderByDesc('is_default')
            ->orderBy('id', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $addresses
        ]);
    }

    public function makeAddressDefault($id)
    {
        $customer = auth('customer')->user();

        $address = Address::where('id', $id)
            ->where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->first();

        if (!$address) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found'
            ], 404);
        }

        Address::where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->update(['is_default' => 0]);

        $address->update(['is_default' => 1]);

        return response()->json([
            'success' => true,
            'message' => 'Address set as default'
        ]);
    }

    public function deleteAddress($id)
    {
        $customer = auth('customer')->user();

        $address = Address::where('id', $id)
            ->where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->first();

        if (!$address) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found'
            ], 404);
        }

        $wasDefault = $address->is_default;

        $address->delete();

        if ($wasDefault) {
            $newDefault = Address::where('user_type', 'customer')
                ->where('user_id', $customer->id)
                ->orderBy('id', 'desc')
                ->first();

            if ($newDefault) {
                $newDefault->update(['is_default' => 1]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Address deleted successfully'
        ]);
    }

    public function getAddress($id)
    {
        $customer = auth('customer')->user();

        $address = Address::where('id', $id)
            ->where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->first();

        if (!$address) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $address
        ]);
    }

    public function updateAddress(Request $request, $id)
    {
        $customer = auth('customer')->user();

        $address = Address::where('id', $id)
            ->where('user_type', 'customer')
            ->where('user_id', $customer->id)
            ->first();

        if (!$address) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found'
            ], 404);
        }

        $validated = $request->validate([
            'name'           => 'required|string|max:255',
            'mobile_number'  => 'required|digits:10',
            'door_no'        => 'nullable|string|max:255',
            'street_name'    => 'nullable|string|max:255',
            'city'           => 'nullable|string|max:255',
            'district_id'    => 'nullable|integer',
            'state_id'       => 'nullable|integer',
            'pincode'        => 'nullable|string|max:20',
            'latitude'       => 'nullable|numeric',
            'longitude'      => 'nullable|numeric',
            'is_default'     => 'nullable|boolean',
        ]);

        /* ---------------- DEFAULT ADDRESS LOGIC ---------------- */

        if ($request->filled('is_default') && $request->is_default) {
            Address::where('user_type', 'customer')
                ->where('user_id', $customer->id)
                ->where('id', '!=', $address->id)
                ->update(['is_default' => 0]);

            $validated['is_default'] = 1;
        }

        /* ---------------- LAT / LON LOGIC ---------------- */

        $latitude  = $request->latitude ?? $address->latitude;
        $longitude = $request->longitude ?? $address->longitude;

        // If lat/lon NOT sent → regenerate from address
        if (!$request->latitude || !$request->longitude) {

            $stateName = State::where('id', $request->state_id ?? $address->state_id)->value('name');
            $districtName = District::where('id', $request->district_id ?? $address->district_id)->value('name');

            $fullAddress = implode(', ', array_filter([
                $request->door_no ?? $address->door_no,
                $request->street_name ?? $address->street_name,
                $request->city ?? $address->city,
                $districtName,
                $stateName,
                $request->pincode ?? $address->pincode,
                'India'
            ]));

            $geo = GeoService::getLatLongFromAddress($fullAddress);

            $latitude  = $geo['latitude'];
            $longitude = $geo['longitude'];
        }

        /* ---------------- UPDATE ADDRESS ---------------- */

        $address->update([
            'name'          => $validated['name'],
            'mobile_number' => $validated['mobile_number'],
            'door_no'       => $validated['door_no'] ?? null,
            'street_name'   => $validated['street_name'] ?? null,
            'city'          => $validated['city'] ?? null,
            'district_id'   => $validated['district_id'] ?? null,
            'state_id'      => $validated['state_id'] ?? null,
            'pincode'       => $validated['pincode'] ?? null,
            'latitude'      => $latitude,
            'longitude'     => $longitude,
            'is_default'    => $validated['is_default'] ?? $address->is_default,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Address updated successfully',
            'data'    => $address->fresh()
        ]);
    }

    
    public function notifications()
    {
        if (!auth('customer')->check()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $customerId = auth('customer')->id();

        $notifications = DB::table('notifications')
            ->where('user_type', 'customer')
            ->where('user_id', $customerId)
            ->whereDate('created_at', '>=', now()->subDays(15))
            ->orderBy('created_at', 'desc')
            ->get();

        $grouped = [];

        foreach ($notifications as $n) {

            $createdAt = Carbon::parse($n->created_at);
            $link = $n->link ? json_decode($n->link, true) : null;

            if ($createdAt->isToday()) {
                $groupKey = 'Today';
            } elseif ($createdAt->isYesterday()) {
                $groupKey = 'Yesterday';
            } else {
                $groupKey = $createdAt->format('Y-m-d');
            }

            $grouped[$groupKey][] = [
                'id'       => $n->id,
                'title'    => $n->title,
                'content'  => $n->content,
                'route'    => $link['route'] ?? null,
                'route_id' => $link['route_id'] ?? null,
                'status'   => $n->status,
                'time'     => $createdAt->format('h:i A'),
            ];
        }

        return response()->json([
            'success' => true,
            'days'    => count($grouped),
            'data'    => $grouped
        ]);
    }

    public function activeNotificationsCount(Request $request)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $activeCount = DB::table('notifications')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('status', 'active')
            ->count();

        return response()->json([
            'success' => true,
            'active_count' => $activeCount
        ]);
    }

    public function toggleNotificationStatus(Request $request, $notificationId)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $notification = DB::table('notifications')
            ->where('id', $notificationId)
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->first();

        if ($notification->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Notification is already inactive'
            ], 400);
        }

        DB::table('notifications')
            ->where('id', $notificationId)
            ->update([
                'status'     => 'inactive',
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Notification deactivated successfully',
            'data' => [
                'notification_id' => $notificationId,
                'status' => 'inactive',
            ]
        ], 200);

    }
}
