<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\OrderRiderRequest;
use App\Models\Store;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;
use Razorpay\Api\Api;
use App\Models\Address;
use App\Models\CartItem;
use App\Helpers\PriceHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\OrderPlacedMail;
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\State;
use App\Models\District;
use App\Services\GeoService;




class StorePurchaseController extends Controller
{
    public function purchaseMainPage(Request $request)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $categories = \DB::table('products as p')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->whereNotNull('p.category_id')
            ->where('p.status', 'active')
            ->select('p.category_id', 'c.name as category', 'c.image as category_image')
            ->groupBy('p.category_id', 'c.name', 'c.image')
            ->orderBy('c.name')
            ->limit(8)
            ->get()
            ->map(function($cat) {
                return [
                    'id' => $cat->category_id,
                    'category' => $cat->category,
                    'image' => $cat->category_image ? asset($cat->category_image) : null
                ];
            });

        $filterCategories = DB::table('products as p')
            ->join('categories as c', 'c.id', '=', 'p.category_id')
            ->whereNotNull('p.category_id')
            ->where('p.status', 'active')
            ->groupBy('p.category_id', 'c.name')
            ->orderBy('c.name')
            ->select(
                'p.category_id as id',
                'c.name'
            )
            ->get();

        $popularProductIds = DB::table('order_items as oi')
            ->join('orders as o', 'o.id', '=', 'oi.order_id')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->where('p.status', 'active')
            ->whereIn('o.status', ['pending_assignment', 'in_transit', 'completed'])
            ->groupBy('oi.product_id')
            ->orderByRaw('COUNT(oi.product_id) DESC')
            ->limit(10)
            ->pluck('oi.product_id')
            ->toArray();

        // If less than 10 → fill with random products
        $needed = 10 - count($popularProductIds);

        if ($needed > 0) {
            $randomIds = DB::table('products')
                ->where('status', 'active')
                ->whereNotIn('id', $popularProductIds)
                ->inRandomOrder()
                ->limit($needed)
                ->pluck('id')
                ->toArray();

            $popularProductIds = array_merge($popularProductIds, $randomIds);
        }

        $popularProducts = DB::table('products as p')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->whereIn('p.id', $popularProductIds)
            ->where('p.status', 'active')
            ->select(
                'p.id',
                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value',
                'p.image',
                'p.colors',
                'c.name as category',
                's.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit'
            )
            ->get()
            ->map(function ($item) {

                $storeDiscounted = $item->store_price;
                if ($item->store_discount_type === 'percentage' && $item->store_discount_value) {
                    $storeDiscounted = $item->store_price - ($item->store_price * $item->store_discount_value / 100);
                } elseif ($item->store_discount_type === 'amount' && $item->store_discount_value) {
                    $storeDiscounted = $item->store_price - $item->store_discount_value;
                }

                // $customerDiscounted = $item->customer_price;
                // if ($item->customer_discount_type === 'percentage' && $item->customer_discount_value) {
                //     $customerDiscounted = $item->customer_price - ($item->customer_price * $item->customer_discount_value / 100);
                // } elseif ($item->customer_discount_type === 'amount' && $item->customer_discount_value) {
                //     $customerDiscounted = $item->customer_price - $item->customer_discount_value;
                // }

                $avgRating = \DB::table('product_reviews')
                    ->where('product_id', $item->id)
                    ->avg('rating');
                $avgRating = $avgRating ? round($avgRating, 1) : 0;

                return [
                    'id'                    => $item->id,
                    'category'              => $item->category,
                    'subcategory'           => $item->subcategory,
                    'brand'                 => $item->brand,
                    'quantity_unit'         => $item->quantity_unit,
                    'store_price'           => $item->store_price,
                    'store_discount_type'   => $item->store_discount_type,
                    'store_discount_value'  => $item->store_discount_value,
                    'store_discounted_price'=> round($storeDiscounted, 2),
                    // 'customer_price'        => $item->customer_price,
                    // 'customer_discount_type'=> $item->customer_discount_type,
                    // 'customer_discount_value'=> $item->customer_discount_value,
                    // 'customer_discounted_price'=> round($customerDiscounted, 2),
                    'image'                 => $item->image ? asset($item->image) : null,
                    'colors'                => $item->colors ? json_decode($item->colors, true) : [],
                    'average_rating'        => $avgRating
                ];
            });

        return response()->json([
            'success'    => true,
            'categories' => $categories,
            'filter_categories'=> $filterCategories,
            'products'   => $popularProducts,
        ], 200);
    }


    public function allCategories()
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $categories = \DB::table('products as p')
            ->join('categories as c', 'c.id', '=', 'p.category_id')
            ->where('p.status', 'active')
            ->select(
                'c.id',
                'c.name',
                'c.image as category_image'  // Added category image
            )
            ->distinct()
            ->orderBy('c.name')
            ->get()
            ->map(function($cat) {
                return [
                    'id' => $cat->id,
                    'name' => $cat->name,
                    'image' => $cat->category_image ? asset($cat->category_image) : null
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $categories
        ]);
    }

    public function productFiltersByCategory($category_id)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        /* ================= SUBCATEGORIES ================= */

        $subcategories = DB::table('products as p')
            ->join('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->where('p.status', 'active')
            ->where('p.category_id', $category_id)
            ->whereNotNull('p.subcategory_id')
            ->groupBy('s.id', 's.name')
            ->orderBy('s.name')
            ->select('s.id', 's.name')
            ->get();

        /* ================= BRANDS ================= */

        $brands = DB::table('products as p')
            ->join('brands as b', 'b.id', '=', 'p.brand_id')
            ->where('p.status', 'active')
            ->where('p.category_id', $category_id)
            ->whereNotNull('p.brand_id')
            ->groupBy('b.id', 'b.name')
            ->orderBy('b.name')
            ->select('b.id', 'b.name')
            ->get();

        /* ================= QUANTITY UNITS ================= */

        $quantityUnits = DB::table('products as p')
            ->join('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('p.status', 'active')
            ->where('p.category_id', $category_id)
            ->whereNotNull('p.quantity_unit_id')
            ->groupBy('q.id', 'q.name')
            ->orderBy('q.name')
            ->select('q.id', 'q.name')
            ->get();

        return response()->json([
            'success' => true,
            'filters' => [
                'subcategories'  => $subcategories,
                'brands'         => $brands,
                'quantity_units' => $quantityUnits
            ]
        ]);
    }

    public function productsByCategory(Request $request, $category_id)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $productsQuery = \DB::table('products as p')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('p.status', 'active')
            ->where('p.category_id', $category_id);

        /* ================= FILTERS ================= */

        if ($request->filled('subcategory_id')) {
            $productsQuery->where('p.subcategory_id', $request->subcategory_id);
        }

        if ($request->filled('brand_id')) {
            $productsQuery->where('p.brand_id', $request->brand_id);
        }

        if ($request->filled('quantity_unit_id')) {
            $productsQuery->where('p.quantity_unit_id', $request->quantity_unit_id);
        }

        /* =========================================== */

        $products = $productsQuery
            ->orderBy('p.created_at', 'desc')
            ->select(
                'p.id',
                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value',
                'p.customer_price',
                'p.customer_discount_type',
                'p.customer_discount_value',
                'p.image',
                'p.colors',
                'c.name as category',
                's.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit'
            )
            ->get()
            ->map(function ($item) {

                $storeDiscounted = $item->store_price;
                $storeDiscountPercentage = null;

                if ($item->store_discount_type === 'percentage' && $item->store_discount_value) {
                    $storeDiscounted -= ($item->store_price * $item->store_discount_value / 100);
                    $storeDiscountPercentage = '-' . round($item->store_discount_value) . '%';
                } elseif ($item->store_discount_type === 'amount' && $item->store_discount_value) {
                    $storeDiscounted -= $item->store_discount_value;
                    if ($item->store_price > 0) {
                        $storeDiscountPercentage = '-' . round(($item->store_discount_value / $item->store_price) * 100) . '%';
                    }
                }

                $avgRating = \DB::table('product_reviews')
                    ->where('product_id', $item->id)
                    ->avg('rating');

                return [
                    'id'                        => $item->id,
                    'category'                  => $item->category,
                    'subcategory'               => $item->subcategory,
                    'brand'                     => $item->brand,
                    'quantity_unit'             => $item->quantity_unit,
                    'store_discounted_price'    => round($storeDiscounted, 2),
                    'store_discount_percentage' => $storeDiscountPercentage,
                    'image'                     => $item->image ? asset($item->image) : null,
                    'colors'                    => $item->colors ? json_decode($item->colors, true) : [],
                    'average_rating'            => $avgRating ? round($avgRating, 1) : 0
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $products
        ]);
    }

    public function productDetail($id)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $product = \DB::table('products as p')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('p.id', $id)
            ->where('p.status', 'active')
            ->select(
                'p.id',
                'p.category_id',
                'p.subcategory_id',
                'p.brand_id',
                'p.quantity_unit_id',
                'p.description',
                'p.colors',
                'p.store_price',
                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value',
                'p.image',
                'c.name as category',
                's.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit'
            )
            ->first();


        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        $reviews = \DB::table('product_reviews as r')
            ->leftJoin('customers as cu', 'cu.id', '=', 'r.customer_id')
            ->leftJoin('stores as st', 'st.id', '=', 'r.store_id')
            ->where('r.product_id', $id)
            ->orderBy('r.created_at', 'desc')
            ->limit(5)
            ->select(
                'r.rating',
                'r.review',
                'r.created_at',
                \DB::raw('COALESCE(cu.name, st.owner_name) as reviewer_name')
            )
            ->get();

        $quantityVariants = \DB::table('products as p')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('p.status', 'active')
            ->where('p.category_id', $product->category_id)
            ->where('p.subcategory_id', $product->subcategory_id)
            ->where('p.brand_id', $product->brand_id)
            ->select(
                'p.id',
                'q.name as quantity_unit',
                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value'
            )
            ->orderBy('q.id')
            ->get();

        $quantityVariants = $quantityVariants->map(function ($item) {

            $discountedPrice = (float) $item->store_price;

            if ($item->store_discount_type && $item->store_discount_type) {
                if ($item->store_discount_type === 'percentage') {
                    $discountedPrice -= ($item->store_price * $item->store_discount_type / 100);
                } else {
                    $discountedPrice -= $item->store_discount_type;
                }
            }

            return [
                'id' => $item->id,
                'quantity_unit' => $item->quantity_unit,
                'store_discounted_price' => round(max($discountedPrice, 0), 2)
            ];
        });


        $storeDiscountedPrice = $product->store_price;
        if ($product->store_discount_type && $product->store_discount_value) {
            if ($product->store_discount_type === 'percentage') {
                $storeDiscountedPrice = $product->store_price - ($product->store_price * $product->store_discount_value / 100);
            } else {
                $storeDiscountedPrice = $product->store_price - $product->store_discount_value;
            }
        }

        $avgRating = \DB::table('product_reviews')
            ->where('product_id', $id)
            ->avg('rating');

        $ratingStats = \DB::table('product_reviews')
            ->where('product_id', $id)
            ->select(
                \DB::raw('COUNT(*) as total_ratings'),
                \DB::raw('SUM(CASE WHEN review IS NOT NULL AND review != "" THEN 1 ELSE 0 END) as total_reviews'),
                \DB::raw('SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as rating_5'),
                \DB::raw('SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as rating_4'),
                \DB::raw('SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as rating_3'),
                \DB::raw('SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as rating_2'),
                \DB::raw('SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as rating_1')
            )
            ->first();

            $relatedProducts = \DB::table('products as p')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('p.status', 'active')
                ->where('p.subcategory_id', $product->subcategory_id)
                ->where('p.id', '!=', $product->id)
                ->select(
                    'p.id',
                    'p.image',
                    'p.store_price',
                    'p.store_discount_type',
                    'p.store_discount_value',
                    'b.name as brand_name',
                    'q.name as quantity_unit'
                )
                ->get();
            
            $relatedCount = $relatedProducts->count();
            if ($relatedCount < 10) {
                $needed = 10 - $relatedCount;
            
                $randomProducts = \DB::table('products as p')
                    ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                    ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                    ->where('p.status', 'active')
                    ->where('p.id', '!=', $product->id)
                    ->whereNotIn('p.id', $relatedProducts->pluck('id'))
                    ->select(
                        'p.id',
                        'p.image',
                        'p.store_price',
                        'p.store_discount_type',
                        'p.store_discount_value',
                        'b.name as brand_name',
                        'q.name as quantity_unit'
                    )
                    ->inRandomOrder()
                    ->limit($needed)
                    ->get();
            
                $relatedProducts = $relatedProducts->merge($randomProducts);
            }
            
            $relatedProducts = $relatedProducts->map(function ($p) {
            
                $originalPrice = (float) $p->store_price;
                $discountedPrice = $originalPrice;
                $discountPercentage = null;
            
                if ($p->store_discount_type === 'percentage') {
                    $discountPercentage = $p->store_discount_value;
                    $discountedPrice = $originalPrice - (($originalPrice * $discountPercentage) / 100);
                } elseif ($p->store_discount_type === 'amount') {
                    $discountAmount = $p->store_discount_value;
                    $discountedPrice = $originalPrice - $discountAmount;
                    if ($originalPrice > 0) {
                        $discountPercentage = ($discountAmount / $originalPrice) * 100;
                    }
                }
            
                $avgRating = \DB::table('product_reviews')
                    ->where('product_id', $p->id)
                    ->where('status', 'active')
                    ->avg('rating');
            
                return [
                    'id' => $p->id,
                    'image' => $p->image ? asset($p->image) : null,
                    'brand_name' => $p->brand_name,
                    'quantity_unit' => $p->quantity_unit,
                    'price' => round($originalPrice, 2),
                    'discounted_price' => round(max($discountedPrice, 0), 2),
                    'discount_percentage' => $discountPercentage ? '-' . round($discountPercentage) . '%' : null,
                    'avg_rating' => $avgRating ? round($avgRating, 1) : 0
                ];
            });


        return response()->json([
            'success' => true,
            'data' => [
                'id'            => $product->id,
                'category'      => $product->category,
                'subcategory'   => $product->subcategory,
                'brand'         => $product->brand,
                'quantity_unit' => $product->quantity_unit,
                'description'   => $product->description,
                'store_price'   => $product->store_price,
                'store_discount_type' => $product->store_discount_type,
                'store_discount_value'=> $product->store_discount_value,
                'store_discounted_price' => round($storeDiscountedPrice, 2),
                'image'         => $product->image ? asset($product->image) : null,
                'colors'        => $product->colors
                                    ? json_decode($product->colors, true)
                                    : [],
                'quantity_variants' => $quantityVariants,
                'average_rating'=> round($avgRating, 1),
                'reviews'       => $reviews,
                'rating_stats'      => [
                    'total_ratings' => $ratingStats->total_ratings ?? 0,
                    'total_reviews' => $ratingStats->total_reviews ?? 0,
                    'rating_5'      => $ratingStats->rating_5 ?? 0,
                    'rating_4'      => $ratingStats->rating_4 ?? 0,
                    'rating_3'      => $ratingStats->rating_3 ?? 0,
                    'rating_2'      => $ratingStats->rating_2 ?? 0,
                    'rating_1'      => $ratingStats->rating_1 ?? 0,
                ],
                'related_products'  => $relatedProducts
            ]
        ], 200);
    }

    public function allReviews($id)
    {
        $store = auth('store')->user();
        if (!$store) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
        }

        $reviews = \DB::table('product_reviews as r')
            ->leftJoin('customers as cu', 'cu.id', '=', 'r.customer_id')
            ->leftJoin('stores as st', 'st.id', '=', 'r.store_id')
            ->where('r.product_id', $id)
            ->orderBy('r.created_at', 'desc')
            ->limit(50) 
            ->select(
                'r.rating',
                'r.review',
                'r.created_at',
                \DB::raw('COALESCE(cu.name, st.store_name) as reviewer_name')
            )
            ->get();

        $avgRating = \DB::table('product_reviews')
            ->where('product_id', $id)
            ->avg('rating');

        $ratingStats = \DB::table('product_reviews')
            ->where('product_id', $id)
            ->select(
                \DB::raw('COUNT(*) as total_ratings'),
                \DB::raw('SUM(CASE WHEN review IS NOT NULL AND review != "" THEN 1 ELSE 0 END) as total_reviews'),
                \DB::raw('SUM(CASE WHEN rating = 5 THEN 1 ELSE 0 END) as rating_5'),
                \DB::raw('SUM(CASE WHEN rating = 4 THEN 1 ELSE 0 END) as rating_4'),
                \DB::raw('SUM(CASE WHEN rating = 3 THEN 1 ELSE 0 END) as rating_3'),
                \DB::raw('SUM(CASE WHEN rating = 2 THEN 1 ELSE 0 END) as rating_2'),
                \DB::raw('SUM(CASE WHEN rating = 1 THEN 1 ELSE 0 END) as rating_1')
            )
            ->first();


        return response()->json([
            'success' => true,
            'average_rating'=> round($avgRating, 1),
                'reviews'       => $reviews,
                'rating_stats'      => [
                    'total_ratings' => $ratingStats->total_ratings ?? 0,
                    'total_reviews' => $ratingStats->total_reviews ?? 0,
                    'rating_5'      => $ratingStats->rating_5 ?? 0,
                    'rating_4'      => $ratingStats->rating_4 ?? 0,
                    'rating_3'      => $ratingStats->rating_3 ?? 0,
                    'rating_2'      => $ratingStats->rating_2 ?? 0,
                    'rating_1'      => $ratingStats->rating_1 ?? 0,
                ],
        ]);
    }

    public function addToCart(Request $request)
    {
        $store = auth('store')->user();
        $customer = auth('customer')->user();

        if (!$store && !$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity'   => 'required|integer|min:1'
        ]);

        $cartItem = CartItem::where('product_id', $request->product_id)
            ->where('store_id', $store?->id)
            ->where('customer_id', $customer?->id)
            ->first();

        if ($cartItem) {
            // Increase quantity
            $cartItem->quantity += $request->quantity;
            $cartItem->save();
        } else {
            CartItem::create([
                'store_id'    => $store?->id,
                'customer_id' => $customer?->id,
                'product_id'  => $request->product_id,
                'quantity'    => $request->quantity
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Product added to cart'
        ]);
    }

    public function removeFromCart(Request $request, $product_id)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $cartItem = DB::table('cart_items')
            ->where('store_id', $store->id)
            ->where('product_id', $product_id)
            ->first();

        if (!$cartItem) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found in cart'
            ], 404);
        }

        DB::table('cart_items')
            ->where('id', $cartItem->id)
            ->delete();

        return response()->json([
            'success' => true,
            'message' => 'Product removed from cart successfully'
        ]);
    }

    public function updateCartQuantity(Request $request, $cartItemId)
    {
        $store = auth('store')->user();
        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $cartItem = CartItem::where('id', $cartItemId)
            ->where('store_id', $store->id)
            ->first();

        if (!$cartItem) {
            return response()->json([
                'success' => false,
                'message' => 'Cart item not found'
            ], 404);
        }

        $action = $request->input('action');

        if ($action === 'increment') {
            $cartItem->quantity += 1;
        } elseif ($action === 'decrement') {
            $cartItem->quantity = max(1, $cartItem->quantity - 1);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Invalid action'
            ], 400);
        }

        $cartItem->save();

        return response()->json([
            'success' => true,
            'message' => 'Cart updated successfully',
            'cart_item' => $cartItem
        ]);
    }

    public function getCart()
    {
        $store = auth('store')->user();
        $customer = auth('customer')->user();

        $cartItems = CartItem::with([
            'product:id,category_id,subcategory_id,brand_id,quantity_unit_id,description,store_price,store_discount_type,store_discount_value,customer_price,customer_discount_type,customer_discount_value,image,colors',
            'product.category:id,name',
            'product.subcategory:id,name',
            'product.brand:id,name',
            'product.quantityUnit:id,name',
        ])
        ->where('store_id', $store?->id)
        ->where('customer_id', $customer?->id)
        ->get();
        
        if ($cartItems->isEmpty()) {
            return response()->json([
                'success' => true,
                'message' => 'Cart is empty',
                'data' => [],
                'addresses' => [],
                'summary' => [
                    'total_products' => 0,
                    'total_price' => 0,
                    'total_discount' => 0,
                    'pending_penalty' => 0,
                    'delivery_charge' => 0,
                    'total_payable_without_delivery' => 0,
                    'total_payable_with_delivery' => 0,
                ]
            ]);
        }


        $totalProducts = $cartItems->count();
        $totalPrice = 0;
        $totalDiscount = 0;

        $calcDiscount = function($price, $type, $value) {
            if (!$type || !$value) return $price;
            if ($type === 'percentage') return $price - ($price * $value / 100);
            if ($type === 'amount') return $price - $value;
            return $price;
        };

        $cart = $cartItems->map(function ($item) use ($store, $customer, $calcDiscount, &$totalPrice, &$totalDiscount) {
            $product = $item->product;

            // $calcDiscount = function($price, $type, $value) {
            //     if (!$type || !$value) return $price;
            //     if ($type === 'percentage') return $price - ($price * $value / 100);
            //     if ($type === 'amount') return $price - $value;
            //     return $price;
            // };

            $storeDiscounted = $calcDiscount($product->store_price, $product->store_discount_type, $product->store_discount_value);
            $customerDiscounted = $calcDiscount($product->customer_price, $product->customer_discount_type, $product->customer_discount_value);

            $unitPrice = auth('store')->check() ? $storeDiscounted : $customerDiscounted;
            $originalUnitPrice = auth('store')->check() ? $product->store_price : $product->customer_price;

            $totalPrice += $originalUnitPrice * $item->quantity;
            $totalDiscount += ($originalUnitPrice - $unitPrice) * $item->quantity;

            $quantityVariants = \DB::table('products as p')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('p.status', 'active')
                ->where('p.category_id', $product->category_id)
                ->where('p.subcategory_id', $product->subcategory_id)
                ->where('p.brand_id', $product->brand_id)
                ->select('p.id', 'q.name as quantity_unit', 'p.store_price', 'p.store_discount_type', 'p.store_discount_value', 'p.customer_price', 'p.customer_discount_type', 'p.customer_discount_value')
                ->orderBy('q.id')
                ->get()
                ->map(function ($q) use ($calcDiscount) {
                    $storeDisc = $calcDiscount($q->store_price, $q->store_discount_type, $q->store_discount_value);
                    $customerDisc = $calcDiscount($q->customer_price, $q->customer_discount_type, $q->customer_discount_value);

                    return [
                        'id' => $q->id,
                        'quantity_unit' => $q->quantity_unit,
                        'store_price' => $q->store_price,
                        'store_discounted' => number_format((float) $storeDisc, 2, '.', ''),
                        // 'customer_price' => $q->customer_price,
                        // 'customer_discounted' => round($customerDisc, 2),
                    ];
                });

            return [
                'cart_item_id'       => $item->id,
                'product_id'         => $product->id,
                'category'           => $product->category?->name,
                'subcategory'        => $product->subcategory?->name,
                'brand'              => $product->brand?->name,
                'quantity_unit'      => $product->quantityUnit?->name,
                'image'              => $product->image ? asset($product->image) : null,
                'colors'             => $product->colors ?? [],
                'quantity_variants'  => $quantityVariants,
                'quantity'           => $item->quantity,
                'store_price'        => $product->store_price,
                'store_discounted'   => round($storeDiscounted, 2),
                // 'customer_price'     => $product->customer_price,
                // 'customer_discounted'=> round($customerDiscounted, 2),
                'total_price'        => round($storeDiscounted * $item->quantity, 2),
            ];
        });

        $pendingPenalty = null;
        if ($store) {
            $pendingPenalty = DB::table('stores')->where('id', $store->id)->value('pending_penalty');
        } elseif ($customer) {
            $pendingPenalty = DB::table('customers')->where('id', $customer->id)->value('pending_penalty');
        }
        $pendingPenalty = $pendingPenalty > 0 ? round($pendingPenalty, 2) : null;

        // Delivery charges
        $defaultDeliveryPrice = DB::table('delivery_prices')->where('is_default', 1)->first();
        $deliveryCharge = auth('store')->check()
            ? ($defaultDeliveryPrice?->store_delivery_price ?? 0)
            : ($defaultDeliveryPrice?->customer_delivery_price ?? 0);

        // Total payable amounts
        $totalPayableWithoutDelivery = $totalPrice - $totalDiscount + ($pendingPenalty ?? 0);
        $totalPayableWithDelivery = $totalPayableWithoutDelivery + $deliveryCharge;

        $addresses = collect();

        if ($customer) {
            $addresses = DB::table('addresses')
                ->where('user_type', 'customer')
                ->where('user_id', $customer->id)
                ->orderByDesc('is_default') // default address first
                ->get();
        } elseif ($store) {
            $addresses = DB::table('addresses')
                ->where('user_type', 'store')
                ->where('user_id', $store->id)
                ->orderByDesc('is_default')
                ->get();
        }

        return response()->json ([
            'success' => true,
            'data' => $cart,
            'addresses' => $addresses,
            'summary' => [
                'total_products' => $totalProducts,
                'total_price' => round($totalPrice, 2),
                'total_discount' => round($totalDiscount, 2),
                'pending_penalty' => $pendingPenalty,
                'delivery_charge' => $deliveryCharge,
                'total_payable_without_delivery' => round($totalPayableWithoutDelivery, 2),
                'total_payable_with_delivery' => round($totalPayableWithDelivery, 2),
            ]
        ]);
    }

    public function cartSelection(Request $request)
    {
        $request->validate([
            'with_shipping' => 'required|boolean',
            'address_id'    => 'required_if:with_shipping,true|nullable|integer',
            'rider_id'      => 'required_if:with_shipping,false|nullable|integer',
        ]);

        $store = auth('store')->user();
        $customer = auth('customer')->user();
        $user = $store ?? $customer;

        if (!$user) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
        }

        $userType = $store ? 'store' : 'customer';

        if ($request->with_shipping) {
            $addressExists = DB::table('addresses')
                ->where('id', $request->address_id)
                ->where('user_type', $userType)
                ->where('user_id', $user->id)
                ->exists();

            if (!$addressExists) {
                return response()->json(['success' => false, 'message' => 'Invalid address'], 400);
            }
        }

        if (!$request->with_shipping) {
            $riderExists = DB::table('riders')
                ->where('id', $request->rider_id)
                ->where('status', 'active')
                ->exists();

            if (!$riderExists) {
                return response()->json(['success' => false, 'message' => 'Invalid rider'], 400);
            }
        }

        $cacheToken = Str::uuid()->toString();

        Cache::put(
            "cart_selection_{$cacheToken}",
            [
                'with_shipping' => $request->with_shipping,
                'address_id'    => $request->with_shipping ? $request->address_id : null,
                'rider_id'      => !$request->with_shipping ? $request->rider_id : null,
                'user_id'       => $user->id,
                'user_type'     => $userType,
            ],
            now()->addHour()
        );

        return response()->json([
            'success' => true,
            'cache_token' => $cacheToken
        ]);
    }

    public function getCartReview(Request $request)
    {
        $request->validate([
            'cache_token' => 'required|string',
        ]);

        $store = auth('store')->user();
        $customer = auth('customer')->user();
        $user = $store ?? $customer;

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        // Retrieve cached selection
        $cacheKey = "cart_selection_{$request->cache_token}";
        $cartSelection = Cache::get($cacheKey);

        if (!$cartSelection) {
            return response()->json([
                'success' => false,
                'message' => 'Cart selection not found or expired'
            ], 404);
        }

        /*
        |--------------------------------------------------------------------------
        | Fetch Delivery Details (Address / Rider)
        |--------------------------------------------------------------------------
        */

        $deliveryDetails = null;

        if ($cartSelection['with_shipping']) {

            // Fetch Address (Verify ownership)
            $deliveryDetails = DB::table('addresses')
                ->where('id', $cartSelection['address_id'])
                ->where('user_id', $user->id)
                ->where('user_type', $store ? 'store' : 'customer')
                ->select(
                    'id',
                    'name',
                    'mobile_number',
                    'door_no',
                    'street_name',
                    'area',
                    'city',
                    'pincode',
                    'latitude',
                    'longitude'
                )
                ->first();

        } else {

            // Fetch Rider Details
            $deliveryDetails = DB::table('riders')
                ->where('id', $cartSelection['rider_id'])
                ->select(
                    'id',
                    'rider_code',
                    'name',
                    'mobile_number'
                )
                ->first();
        }


        // Fetch cart items
        $cartItems = CartItem::with([
            'product:id,category_id,subcategory_id,brand_id,quantity_unit_id,description,store_price,store_discount_type,store_discount_value,customer_price,customer_discount_type,customer_discount_value,image,colors',
            'product.category:id,name',
            'product.subcategory:id,name',
            'product.brand:id,name',
            'product.quantityUnit:id,name',
        ])
        ->where('store_id', $store?->id)
        ->where('customer_id', $customer?->id)
        ->get();

        $totalProducts = $cartItems->count();
        $totalPrice = 0;
        $totalDiscount = 0;

        $calcDiscount = function ($price, $type, $value) {
            if (!$type || !$value) return $price;
            if ($type === 'percentage') return $price - ($price * $value / 100);
            if ($type === 'amount') return $price - $value;
            return $price;
        };

        // Map cart items
        $cart = $cartItems->map(function ($item) use ($calcDiscount, &$totalPrice, &$totalDiscount) {

            $product = $item->product;

            $storeDiscounted = $calcDiscount(
                $product->store_price,
                $product->store_discount_type,
                $product->store_discount_value
            );

            $customerDiscounted = $calcDiscount(
                $product->customer_price,
                $product->customer_discount_type,
                $product->customer_discount_value
            );

            // Decide unit price based on login
            $isStore = auth('store')->check();

            $unitPrice = $isStore ? $storeDiscounted : $customerDiscounted;
            $originalUnitPrice = $isStore ? $product->store_price : $product->customer_price;

            // Accumulate totals
            $totalPrice += $originalUnitPrice * $item->quantity;
            $totalDiscount += ($originalUnitPrice - $unitPrice) * $item->quantity;

            return [
                'cart_item_id' => $item->id,
                'product_id' => $product->id,
                'category' => $product->category?->name,
                'subcategory' => $product->subcategory?->name,
                'brand' => $product->brand?->name,
                'quantity_unit' => $product->quantityUnit?->name,
                'image' => $product->image ? asset($product->image) : null,
                'colors' => $product->colors ?? [],
                'quantity' => $item->quantity,
                'unit_price' => round($unitPrice, 2),
                'original_price' => round($originalUnitPrice, 2),
                'total_price' => round($unitPrice * $item->quantity, 2),
            ];
        });

        // Pending penalty
        $pendingPenalty = null;

        if ($store) {
            $pendingPenalty = DB::table('stores')
                ->where('id', $store->id)
                ->value('pending_penalty');
        } elseif ($customer) {
            $pendingPenalty = DB::table('customers')
                ->where('id', $customer->id)
                ->value('pending_penalty');
        }

        $pendingPenalty = $pendingPenalty > 0 ? round($pendingPenalty, 2) : null;

        // Delivery charges
        $defaultDeliveryPrice = DB::table('delivery_prices')
            ->where('is_default', 1)
            ->first();

        $deliveryCharge = auth('store')->check()
            ? ($defaultDeliveryPrice?->store_delivery_price ?? 0)
            : ($defaultDeliveryPrice?->customer_delivery_price ?? 0);

        // Final totals
        $totalPayableWithoutDelivery = $totalPrice - $totalDiscount + ($pendingPenalty ?? 0);
        $totalPayableWithDelivery = $totalPayableWithoutDelivery + $deliveryCharge;

        return response()->json([
            'success' => true,
            'with_shipping' => (int) $cartSelection['with_shipping'], // 1 or 0
            'delivery_details' => $deliveryDetails, // address or rider
            'data' => $cart,
            'summary' => [
                'total_products' => $totalProducts,
                'total_price' => round($totalPrice, 2),
                'total_discount' => round($totalDiscount, 2),
                'pending_penalty' => $pendingPenalty,
                'delivery_charge' => round($deliveryCharge, 2),
                'total_payable_without_delivery' => round($totalPayableWithoutDelivery, 2),
                'total_payable_with_delivery' => round($totalPayableWithDelivery, 2),
            ]
        ]);
    }


    public function buyNow(Request $request)
    {
        $customer = auth('customer')->user();
        $store    = auth('store')->user();

        if (!$customer && !$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity'   => 'required|integer|min:1'
        ]);

        \DB::table('cart_items')
            ->where('product_id', $request->product_id)
            ->where(function ($q) use ($customer, $store) {
                if ($customer) $q->where('customer_id', $customer->id);
                if ($store)    $q->where('store_id', $store->id);
            })
            ->delete();

        $cartId = \DB::table('cart_items')->insertGetId([
            'customer_id' => $customer?->id,
            'store_id'    => $store?->id,
            'product_id'  => $request->product_id,
            'quantity'    => $request->quantity,
            'created_at'  => now(),
            'updated_at'  => now()
        ]);

        return response()->json([
            'success' => true,
            'cart_item_id' => $cartId,
            'message' => 'Buy Now item added'
        ]);
    }

    public function buyNowReview($productId)
    {
        $customer = auth('customer')->user();
        $store    = auth('store')->user();

        if (!$customer && !$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $cartItem = \DB::table('cart_items as ci')
            ->join('products as p', 'p.id', '=', 'ci.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('ci.product_id', $productId)
            ->where(function ($q) use ($customer, $store) {
                if ($customer) $q->where('ci.customer_id', $customer->id);
                if ($store)    $q->where('ci.store_id', $store->id);
            })
            ->orderBy('ci.id', 'desc')
            ->select(
                'ci.quantity',
                'ci.store_id',

                'p.id as product_id',
                'p.image',
                'p.colors',

                'p.category_id',
                'p.subcategory_id',
                'p.brand_id',

                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value',
                'p.customer_price',
                'p.customer_discount_type',
                'p.customer_discount_value',

                'c.name as category',
                's.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit'
            )
            ->first();

        if (!$cartItem) {
            return response()->json([
                'success' => false,
                'message' => 'Cart item not found'
            ], 404);
        }

        /* ================= Discount Function ================= */
        $calcDiscount = function($price, $type, $value) {
            if (!$type || !$value) return $price;
            if ($type === 'percentage') return $price - ($price * $value / 100);
            if ($type === 'amount') return $price - $value;
            return $price;
        };

        $calcDiscountPercentage = function ($original, $discounted) {
            if ($original <= 0) return null;

            $percentage = (($original - $discounted) / $original) * 100;

            return $percentage > 0 ? round($percentage) : null;
        };

        $storeDiscounted    = $calcDiscount($cartItem->store_price, $cartItem->store_discount_type, $cartItem->store_discount_value);
        $customerDiscounted = $calcDiscount($cartItem->customer_price, $cartItem->customer_discount_type, $cartItem->customer_discount_value);
        $customerDiscountPercentage = $calcDiscountPercentage($cartItem->customer_price,$customerDiscounted);

        /* ================= Quantity Variants ================= */
        $quantityVariants = \DB::table('products as p')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('p.status', 'active')
            ->where('p.category_id', $cartItem->category_id)
            ->where('p.subcategory_id', $cartItem->subcategory_id)
            ->where('p.brand_id', $cartItem->brand_id)
            ->select(
                'p.id',
                'q.name as quantity_unit',
                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value',
                'p.customer_price',
                'p.customer_discount_type',
                'p.customer_discount_value'
            )
            ->orderBy('q.id')
            ->get()
            ->map(function ($q) use ($calcDiscount, $calcDiscountPercentage) {
                return [
                    'id' => $q->id,
                    'quantity_unit' => $q->quantity_unit,

                    'store_price' => $q->store_price,
                    'store_discounted' => round($calcDiscount(
                        $q->store_price,
                        $q->store_discount_type,
                        $q->store_discount_value
                    ), 2),

                    // 'customer_price' => $q->customer_price,
                    // 'customer_discounted' => round($calcDiscount(
                    //     $q->customer_price,
                    //     $q->customer_discount_type,
                    //     $q->customer_discount_value
                    // ), 2),
                ];
            });

        /* ================= Price Calculation ================= */
        $originalUnitPrice = auth('store')->check()
            ? $cartItem->store_price
            : $cartItem->customer_price;

        $discountedUnitPrice = auth('store')->check()
            ? $storeDiscounted
            : $customerDiscounted;

        $productOriginalTotal = $originalUnitPrice * $cartItem->quantity;
        $productDiscountedTotal = $discountedUnitPrice * $cartItem->quantity;

        $discountAmount = $productOriginalTotal - $productDiscountedTotal;

        $defaultDeliveryPrice = DB::table('delivery_prices')
            ->where('is_default', 1)
            ->first();

        $deliveryCharge = auth('store')->check()
            ? ($defaultDeliveryPrice?->store_delivery_price ?? 0)
            : ($defaultDeliveryPrice?->customer_delivery_price ?? 0);

        $deliveryDate = now()->addDays(7)->format('d M Y');

        $penaltyAmount = 0;
        if ($store) {
            $penaltyAmount = DB::table('stores')
                ->where('id', $store->id)
                ->value('pending_penalty');
        } elseif ($customer) {
            $penaltyAmount = DB::table('customers')
                ->where('id', $customer->id)
                ->value('pending_penalty');
        }
        $penaltyAmount = $penaltyAmount > 0 ? round($penaltyAmount, 2) : null;

        // Total payable without delivery
        $totalPayableWithoutDelivery = $productDiscountedTotal;
        if ($penaltyAmount) {
            $totalPayableWithoutDelivery += $penaltyAmount;
        }

        // Total payable with delivery
        $totalPayableWithDelivery = $productDiscountedTotal + $deliveryCharge;
        if ($penaltyAmount) {
            $totalPayableWithDelivery += $penaltyAmount;
        }

        /* ================= Address ================= */
        $storeId = $store?->id;

        $defaultAddress = DB::table('addresses')
            ->where('user_type', 'store')
            ->where('user_id', $storeId)
            ->where('is_default', 1)
            ->first();

        $allAddresses = DB::table('addresses')
            ->where('user_type', 'store')
            ->where('user_id', $storeId)
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'product' => [
                    'product_id'   => $cartItem->product_id,
                    'category'     => $cartItem->category,
                    'subcategory'  => $cartItem->subcategory,
                    'brand'        => $cartItem->brand,
                    'quantity_unit'=> $cartItem->quantity_unit,
                    'image'        => $cartItem->image ? asset($cartItem->image) : null,

                    'colors'       => $cartItem->colors ? json_decode($cartItem->colors, true) : [],

                    'quantity'     => $cartItem->quantity,

                    /* actual prices */
                    'store_price'    => $cartItem->store_price,
                    // 'customer_price' => $cartItem->customer_price,

                    /* discounted prices */
                    'store_discounted'    => round($storeDiscounted, 2),
                    // 'customer_discounted' => round($customerDiscounted, 2),

                    // 'unit_price'   => $unitPrice,
                    // 'total_price'  => $totalProductPrice,

                    'quantity_variants' => $quantityVariants
                ],

                'delivery' => [
                    'delivery_by'     => $deliveryDate,
                    'delivery_charge' => $deliveryCharge
                ],

                'store' => [
                    'owner_name' => $store?->owner_name ?? null,
                    'name'       => $store?->store_name ?? null,
                    'phone'      => $store?->mobile_number ?? null,
                    'address'    => $defaultAddress ? trim(
                        ($defaultAddress->door_no ?? '') . ', ' .
                        ($defaultAddress->street_name ?? '') . ', ' .
                        ($defaultAddress->area ?? '') . ', ' .
                        ($defaultAddress->city ?? '') . ', ' .
                        ($defaultAddress->district_id ? \DB::table('districts')->where('id', $defaultAddress->district_id)->value('name') : '') . ', ' .
                        ($defaultAddress->state_id ? \DB::table('states')->where('id', $defaultAddress->state_id)->value('name') : '') . ' - ' .
                        ($defaultAddress->pincode ?? '')
                    ) : null,

                    'all_addresses' => $allAddresses
                ],

                'price_summary' => [
                    'product_price'       => round($productOriginalTotal, 2),   // 440
                    'discount'            => round($discountAmount, 2),         // 50
                    'delivery_charges'    => $deliveryCharge,   // 100
                    'penalty_amount'      => $penaltyAmount,                
                    'total_payable_without_delivery' => round($totalPayableWithoutDelivery, 2),
                    'total_payable_with_delivery'    => round($totalPayableWithDelivery, 2)          // 490
                ]
            ]
        ]);
    }
    public function addAddress(Request $request)
    {
        $user = auth()->user();
        $userType = null;

        if (auth('store')->check()) $userType = 'store';
        elseif (auth('customer')->check()) $userType = 'customer';
        elseif (auth('rider')->check()) $userType = 'rider';
        elseif (auth('warehouse')->check()) $userType = 'warehouse';
        else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $request->validate([
            'name'          => 'required|string|max:255',
            'mobile_number' => 'required|string|max:20',
            'door_no'       => 'required|string|max:50',
            'street_name'   => 'required|string|max:255',
            'city'          => 'required|string|max:255',
            'district_id'   => 'required|exists:districts,id',
            'state_id'      => 'required|exists:states,id',
            'pincode'       => 'required|string|max:20',
            'latitude'      => 'nullable|numeric',
            'longitude'     => 'nullable|numeric'
        ]);

        // ---------------- LAT / LON LOGIC ----------------

        $latitude  = $request->latitude;
        $longitude = $request->longitude;

        if (!$latitude || !$longitude) {

            $stateName    = State::where('id', $request->state_id)->value('name');
            $districtName = District::where('id', $request->district_id)->value('name');

            $fullAddress = implode(', ', array_filter([
                $request->door_no,
                $request->street_name,
                $request->city,
                $districtName,
                $stateName,
                $request->pincode,
                'India'
            ]));

            $geo = GeoService::getLatLongFromAddress($fullAddress);

            $latitude  = $geo['latitude']  ?? null;
            $longitude = $geo['longitude'] ?? null;
        }

        // ---------------- CREATE ----------------

        $address = Address::create([
            'user_type'     => $userType,
            'user_id'       => $user->id,
            'name'          => $request->name,
            'mobile_number' => $request->mobile_number,
            'door_no'       => $request->door_no,
            'street_name'   => $request->street_name,
            'city'          => $request->city,
            'district_id'   => $request->district_id,
            'state_id'      => $request->state_id,
            'pincode'       => $request->pincode,
            'latitude'      => $latitude,
            'longitude'     => $longitude,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Address added successfully',
            'data'    => $address
        ]);
    }


    public function editAddress(Request $request, $id)
    {
        $user = auth()->user();
        $userType = null;

        if (auth('store')->check()) $userType = 'store';
        elseif (auth('customer')->check()) $userType = 'customer';
        elseif (auth('rider')->check()) $userType = 'rider';
        elseif (auth('warehouse')->check()) $userType = 'warehouse';
        else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $address = Address::where('user_type', $userType)
            ->where('user_id', $user->id)
            ->where('id', $id)
            ->first();

        if (!$address) {
            return response()->json([
                'success' => false,
                'message' => 'Address not found'
            ], 404);
        }

        $request->validate([
            'name'          => 'sometimes|required|string|max:255',
            'mobile_number' => 'sometimes|required|string|max:20',
            'door_no'       => 'sometimes|required|string|max:50',
            'street_name'   => 'sometimes|required|string|max:255',
            'city'          => 'sometimes|required|string|max:255',
            'district_id'   => 'sometimes|required|exists:districts,id',
            'state_id'      => 'sometimes|required|exists:states,id',
            'pincode'       => 'sometimes|required|string|max:20',
            'is_default'    => 'nullable|boolean'
        ]);

        if ($request->has('is_default') && $request->is_default) {
            Address::where('user_type', $userType)
                ->where('user_id', $user->id)
                ->update(['is_default' => 0]);
        }

        $address->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Address updated successfully',
            'data' => $address
        ]);
    }

    public function verifyRiderTemp(Request $request)
    {
        $request->validate([
            'rider_code'    => 'required|exists:riders,rider_code',
            'mobile_number' => 'required'
        ]);

        $store    = auth('store')->user();
        $customer = auth('customer')->user();

        if (!$store && !$customer) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $rider = \DB::table('riders')
            ->where('rider_code', $request->rider_code)
            ->where('mobile_number', $request->mobile_number)
            ->where('status', 'active')
            ->first();

        if (!$rider) {
            return response()->json([
                'success' => false,
                'message' => 'Rider not found or details do not match'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Rider verified',
            'rider' => [
                'id'         => $rider->id,
                'rider_code' => $rider->rider_code,
                'name'       => $rider->name,
                'phone'      => $rider->mobile_number
            ]
        ]);
    }


    public function CartplaceOrder(Request $request)
    {
        $customer = auth('customer')->user();
        $store    = auth('store')->user();

        if (!$customer && !$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $request->validate([
            'use_shipping' => 'required|in:0,1',
            'shipping_address_id' => 'required_if:use_shipping,1|exists:addresses,id',
            'rider_code' => 'required_if:use_shipping,0',
            'rider_mobile_number' => 'required_if:use_shipping,0|digits:10'
        ]);

        $useShipping = (bool) $request->use_shipping;

        $cartItems = DB::table('cart_items as ci')
            ->join('products as p', 'p.id', '=', 'ci.product_id')
            ->where(function ($q) use ($customer, $store) {
                if ($customer) $q->where('ci.customer_id', $customer->id);
                if ($store)    $q->where('ci.store_id', $store->id);
            })
            ->select(
                'ci.quantity',
                'p.id as product_id',

                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value',

                'p.customer_price',
                'p.customer_discount_type',
                'p.customer_discount_value'
            )
            ->get();

        if ($cartItems->isEmpty()) {
            return response()->json([
                'success' => false,
                'message' => 'Cart is empty'
            ], 400);
        }

        $riderId = null;
        $shippingAddressId = null;

        if ($useShipping) {
            $shippingAddressId = $request->shipping_address_id;
        } else {
            $rider = DB::table('riders')
                ->where('rider_code', $request->rider_code)
                ->where('mobile_number', $request->rider_mobile_number)
                ->where('status', 'active')
                ->first();

            if (!$rider) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid rider code or mobile number'
                ], 422);
            }

            $riderId = $rider->id;
        }

        $pendingPenalty = 0;
        if ($customer) {
            $pendingPenalty = $customer->pending_penalty ?? 0;
        } elseif ($store) {
            $pendingPenalty = $store->pending_penalty ?? 0;
        }

        $defaultDelivery = DB::table('delivery_prices')
            ->where('is_default', 1)
            ->first();

        $storeDeliveryCharge    = $defaultDelivery?->store_delivery_price ?? 0;
        $customerDeliveryCharge = $defaultDelivery?->customer_delivery_price ?? 0;


        if ($useShipping) {
            $deliveryCharge = $store
                ? $storeDeliveryCharge
                : $customerDeliveryCharge;
        } else {
            $deliveryCharge = 0;
        }

        $ordersData = [];

        foreach ($cartItems as $item) {

            if ($store) {
                $originalUnitPrice = $item->store_price;
                $discountType  = $item->store_discount_type;
                $discountValue = $item->store_discount_value;
            } else {
                $originalUnitPrice = $item->customer_price;
                $discountType  = $item->customer_discount_type;
                $discountValue = $item->customer_discount_value;
            }

            $discountedUnitPrice = $originalUnitPrice;

            if ($discountType === 'percentage' && $discountValue > 0) {
                $discountedUnitPrice -= ($originalUnitPrice * $discountValue / 100);
            } elseif ($discountType === 'amount' && $discountValue > 0) {
                $discountedUnitPrice -= $discountValue;
            }

            $discountedUnitPrice = max(0, $discountedUnitPrice);

            $originalTotal  = $originalUnitPrice * $item->quantity;
            $finalTotal     = $discountedUnitPrice * $item->quantity;
            $discountAmount = $originalTotal - $finalTotal;

            if ($useShipping) {
                $deliveryCharge = $store
                    ? $storeDeliveryCharge
                    : $customerDeliveryCharge;
            } else {
                $deliveryCharge = 0;
            }
            
            $grandTotal     = $finalTotal + $deliveryCharge;

            $ordersData[] = [
                'order_no' => 'TEMP-' . strtoupper(Str::random(8)),
                'user_type' => $customer ? 'customer' : 'store',
                'user_id' => $customer ? $customer->id : $store->id,

                'with_shipping' => $useShipping,
                'rider_id' => $riderId,
                'shipping_address_id' => $shippingAddressId,

                'items_total' => $originalTotal,
                'discount_amount' => $discountAmount,
                'delivery_fee' => $deliveryCharge,
                'grand_total' => $grandTotal,

                'pending_penalty' => $pendingPenalty > 0 ? $pendingPenalty : null,

                'cart_item' => [
                    'product_id' => $item->product_id,
                    'quantity'   => $item->quantity,

                    'original_unit_price'   => $originalUnitPrice,
                    'discounted_unit_price' => $discountedUnitPrice,
                    'discount_amount'       => $discountAmount,

                    'unit_price'  => $discountedUnitPrice,
                    'total_price' => $finalTotal,
                ]
            ];
        }

        $cacheKey = 'pending_order_' . ($customer ? $customer->id : $store->id) . '_' . Str::random(6);
        Cache::put($cacheKey, $ordersData, now()->addMinutes(20));

        return response()->json([
            'success' => true,
            'message' => 'Order ready for payment',
            'cache_key' => $cacheKey,
            'orders_preview' => $ordersData
        ]);
    }

    public function placeOrder(Request $request)
    {
        $request->validate([
            'product_id'          => 'required|exists:products,id',
            'use_shipping'        => 'required|boolean',
            'shipping_address_id' => 'required_if:use_shipping,true|exists:addresses,id',
            'rider_code'          => 'required_if:use_shipping,false',
            'rider_mobile_number' => 'required_if:use_shipping,false'
        ]);

        $customer = auth('customer')->user();
        $store    = auth('store')->user();

        if (!$customer && !$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $cartItem = DB::table('cart_items as ci')
            ->join('products as p', 'p.id', '=', 'ci.product_id')
            ->where('ci.product_id', $request->product_id)
            ->where(function ($q) use ($customer, $store) {
                if ($customer) $q->where('ci.customer_id', $customer->id);
                if ($store)    $q->where('ci.store_id', $store->id);
            })
            ->select(
                'ci.quantity',
                'p.id as product_id',
                'p.store_price',
                'p.store_discount_type',
                'p.store_discount_value',
                'p.customer_price',
                'p.customer_discount_type',
                'p.customer_discount_value'
            )
            ->first();

        if (!$cartItem) {
            return response()->json([
                'success' => false,
                'message' => 'Cart item not found'
            ], 404);
        }

        if ($store) {
            $originalUnitPrice = $cartItem->store_price;
            $discountType      = $cartItem->store_discount_type;
            $discountValue     = $cartItem->store_discount_value;
        } else {
            $originalUnitPrice = $cartItem->customer_price;
            $discountType      = $cartItem->customer_discount_type;
            $discountValue     = $cartItem->customer_discount_value;
        }

        $discountedUnitPrice = $originalUnitPrice;

        if ($discountType === 'percentage' && $discountValue > 0) {
            $discountedUnitPrice -= ($originalUnitPrice * $discountValue / 100);
        } elseif ($discountType === 'amount' && $discountValue > 0) {
            $discountedUnitPrice -= $discountValue;
        }

        $discountedUnitPrice = max(0, $discountedUnitPrice);

        $originalTotal  = $originalUnitPrice * $cartItem->quantity;
        $finalTotal     = $discountedUnitPrice * $cartItem->quantity;
        $discountAmount = $originalTotal - $finalTotal;

        $riderId = null;
        $shippingAddressId = null;

        if ($request->use_shipping) {
            $shippingAddressId = $request->shipping_address_id;
        } else {
            $rider = DB::table('riders')
                ->where('rider_code', $request->rider_code)
                ->where('mobile_number', $request->rider_mobile_number)
                ->where('status', 'active')
                ->first();

            if (!$rider) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid rider details'
                ], 422);
            }

            $riderId = $rider->id;
        }

        $defaultDelivery = DB::table('delivery_prices')
            ->where('is_default', 1)
            ->first();

        $storeDeliveryCharge    = $defaultDelivery->store_delivery_price ?? 0;
        $customerDeliveryCharge = $defaultDelivery->customer_delivery_price ?? 0;

        if ($request->use_shipping) {
            $deliveryFee = $store
                ? $storeDeliveryCharge
                : $customerDeliveryCharge;
        } else {
            $deliveryFee = 0;
        }

        $grandTotal  = $finalTotal + $deliveryFee;


        $pendingPenalty = 0;
        if ($customer) {
            $pendingPenalty = $customer->pending_penalty ?? 0;
        } elseif ($store) {
            $pendingPenalty = $store->pending_penalty ?? 0;
        }

        $cachedOrder = [[
            'order_no'  => 'TEMP-' . strtoupper(Str::random(8)),
            'user_type' => $customer ? 'customer' : 'store',
            'user_id'   => $customer ? $customer->id : $store->id,

            'with_shipping'       => $request->use_shipping,
            'rider_id'            => $riderId,
            'shipping_address_id' => $shippingAddressId,

            'items_total'     => $originalTotal,
            'discount_amount' => $discountAmount,
            'delivery_fee'    => $deliveryFee,
            'grand_total'     => $grandTotal,
            'pending_penalty' => $pendingPenalty > 0 ? $pendingPenalty : null,

            'cart_item' => [
                'product_id' => $cartItem->product_id,
                'quantity'   => $cartItem->quantity,

                'original_unit_price'   => $originalUnitPrice,
                'discounted_unit_price' => $discountedUnitPrice,
                'discount_amount'       => $discountAmount,

                'unit_price'  => $discountedUnitPrice,
                'total_price' => $finalTotal,
            ]
        ]];


        $cacheKey = 'pending_order_' . ($customer ? $customer->id : $store->id) . '_' . Str::random(6);


        Cache::put($cacheKey, $cachedOrder, now()->addMinutes(20));

        return response()->json([
            'success'   => true,
            'message'   => 'Order ready for payment',
            'cache_key' => $cacheKey,
            'preview'   => $cachedOrder
        ]);
    }

    private function createOrderFromCache(array $cachedOrder, string $paymentMethod, string $paymentStatus, bool $isFirstOrder = false)
    {
        $store = auth('store')->user();

        $paidPending = $isFirstOrder ? ($cachedOrder['pending_penalty'] ?? 0) : 0;

        try {
            $orderId = DB::table('orders')->insertGetId([
                'order_no' => 'ORD-' . strtoupper(Str::random(8)),
                'user_type' => 'store',
                'user_id'   => $store->id,
                'with_shipping' => $cachedOrder['with_shipping'] ?? 0,
                'rider_id' => $cachedOrder['rider_id'] ?? null,
                'shipping_address_id' => $cachedOrder['shipping_address_id'] ?? null,
                'payment_method' => $paymentMethod,
                'payment_status' => $paymentStatus,
                'status' => 'pending_assignment',
                'items_total' => $cachedOrder['items_total'] ?? 0,
                'discount_amount' => $cachedOrder['discount_amount'] ?? 0,
                'delivery_fee' => $cachedOrder['delivery_fee'] ?? 0,
                'grand_total' => $cachedOrder['grand_total'] ?? 0,
                'paid_pending' => $paidPending,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            } catch (\Exception $e) {
            \Log::channel('razorpay_orders')->error('Failed to insert order', [
                'cachedOrder' => $cachedOrder,
                'error' => $e->getMessage(),
            ]);
            return null;
        }

        $items = $cachedOrder['cart_items'] ?? [];
            if (empty($items) && isset($cachedOrder['cart_item'])) {
                $items = [$cachedOrder['cart_item']];
            }

            if (empty($items)) {
                throw new \Exception('No cart items found in cached order');
            }


        foreach ($items as $item) {
            if (!$item) continue;
            DB::table('order_items')->insert([
                'order_id' => $orderId,
                'product_id' => $item['product_id'],
                'quantity' => $item['quantity'],
                'original_unit_price' => $item['original_unit_price'],
                'discounted_unit_price' => $item['discounted_unit_price'],
                'discount_amount' => $item['discount_amount'],
                'unit_price' => $item['unit_price'],
                'total_price' => $item['total_price'],
                'status' => 'pending',
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }
        return $orderId;
    }

    /**
     * Send order placed email to the user (store or customer) if email exists
     *
     * @param  \stdClass|\App\Models\Order  $order
     * @return void
     */
    private function sendOrderPlacedEmail($order)
    {
        if (!$order) return;

        if ($order instanceof \stdClass) {
            $order = Order::find($order->id);
            if (!$order) return;
        }

        $order->item = DB::table('order_items as oi')
            ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->where('oi.order_id', $order->id)
            ->select(
                'oi.quantity',
                'oi.total_price',
                'p.brand',
                 DB::raw('CASE WHEN oi.quantity > 0 THEN oi.total_price / oi.quantity ELSE 0 END as unit_price'),
                'p.category',
                'p.subcategory',
                'p.image',
                'p.quantity_unit',
                's.name as subcategory_name'
            )
            ->first();

        if (!$order->item) return;

        $user = null;
        if ($order->user_type === 'store') {
            $user = Store::find($order->user_id);
        } elseif ($order->user_type === 'customer') {
            $user = Customer::find($order->user_id);
        }

        if ($user && !empty($user->email)) {
            Mail::to($user->email)->send(new OrderPlacedMail($order, $user));
        }
    }

    private function generateUniqueOrderNo()
    {
        do {
            $orderNo = 'ORD-' . strtoupper(Str::random(8));
        } while (Order::where('order_no', $orderNo)->exists());

        return $orderNo;
    }

    private function generateInvoiceNo()
    {
        $datePrefix = date('Ymd');
        $lastInvoice = Order::where('invoice_no', 'like', "INV-$datePrefix%")
                        ->orderBy('invoice_no', 'desc')
                        ->first();

        $increment = 1;
        if ($lastInvoice) {
            $lastNumber = intval(substr($lastInvoice->invoice_no, -4));
            $increment = $lastNumber + 1;
        }

        return 'INV-' . $datePrefix . '-' . str_pad($increment, 4, '0', STR_PAD_LEFT);
    }

    public function PaymentChoose(Request $request)
    {
        $request->validate([
            'payment_method' => 'required|in:COD,Razorpay,Cashfree',
            'cache_key'      => 'required'
        ]);

        $store = auth('store')->user();
        if (!$store) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
        }

        if (!Cache::has($request->cache_key)) {
            return response()->json(['success' => false, 'message' => 'Order session expired'], 410);
        }

        $cachedOrders = Cache::get($request->cache_key);

        if ($request->payment_method === 'COD') {
            try {
                DB::beginTransaction();
                $createdOrderIds = [];
                $firstOrder = true;

                foreach ($cachedOrders as $cachedOrder) {
                    if (!$cachedOrder) continue;

                    $orderId = $this->createOrderFromCache($cachedOrder, 'COD', 'unpaid', $firstOrder);
                    if (!$orderId) continue;

                    $orderObj = DB::table('orders')->where('id', $orderId)->first();

                    if ($orderObj) {
                        $this->notifyNearbyWarehousesWithProducts($orderObj, 10);
                        $this->sendOrderPlacedEmail($orderObj);
                    }                    
                    $createdOrderIds[] = $orderId;
                    $firstOrder = false;
                }

                 DB::table('stores')->where('id', $store->id)->update(['pending_penalty' => 0]);

                Cache::forget($request->cache_key);
                DB::table('cart_items')->where('store_id', $store->id)->delete();
                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'COD order placed successfully',
                    'order_ids' => $createdOrderIds
                ]);

            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
            }
        }

        $paymentRef = 'PAY-' . Str::uuid();
        Cache::put('payment_ref_'.$paymentRef, $request->cache_key, now()->addMinutes(30));

        return response()->json([
            'success' => true,
            'message' => 'Proceed to payment gateway',
            'payment_ref' => $paymentRef,
            'payment_method' => $request->payment_method
        ]);
    }

    public function createRazorpayPaymentLink(Request $request)
    {
        $request->validate(['cache_key' => 'required']);
        $store = auth('store')->user();
        if (!$store) return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
        if (!Cache::has($request->cache_key)) return response()->json(['success' => false, 'message' => 'Order session expired'], 410);

        $cachedOrders = Cache::get($request->cache_key);
        if (!$cachedOrders || !is_array($cachedOrders)) return response()->json(['success' => false, 'message' => 'No orders in cache'], 410);

        try {
            $api = new Api(config('services.razorpay.key'), config('services.razorpay.secret'));

            $totalAmount = 0;
            $pendingPenalty = 0;

            foreach ($cachedOrders as $index => $order) {
                // Sum all grand totals
                $totalAmount += $order['grand_total'] ?? 0;

                if ($index === 0) {
                    $pendingPenalty = $order['pending_penalty'] ?? 0;
                }
            }

            $amountToPay = $totalAmount + $pendingPenalty;

            $firstOrder = $cachedOrders[0];

            $paymentLink = $api->paymentLink->create([
                'amount' => $amountToPay  * 100,
                'currency' => 'INR',
                'accept_partial' => false,
                'description' => 'Payment for '.count($cachedOrders).' orders',
                'reference_id' => $request->cache_key,
                'customer' => [
                    'name' => $firstOrder['customer_name'] ?? $store->name,
                    'email' => $firstOrder['customer_email'] ?? null,
                    'contact' => $firstOrder['customer_mobile'] ?? null,
                ],
                'notify' => ['sms' => true, 'email' => true],
                'callback_url' => route('razorpay.payment.success'),
                'callback_method' => 'get',
                'notes' => ['store_id' => $store->id]
            ]);

            return response()->json([
                'success' => true,
                'payment_url' => $paymentLink['short_url'],
                'payment_link_id' => $paymentLink['id']
            ]);

        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function razorpayPaymentSuccess(Request $request)
    {
        \Log::info('Razorpay callback hit', $request->all());

        if ($request->razorpay_payment_link_status !== 'paid') {
            return response()->json([
                'success' => false,
                'message' => 'Payment not completed'
            ], 400);
        }

        $cacheKey = $request->razorpay_payment_link_reference_id;

        $cachedOrders = Cache::get($cacheKey);

        if (!$cachedOrders || empty($cachedOrders)) {
            \Log::error('No orders in cached data', ['cacheKey' => $cacheKey]);

            return response()->json([
                'success' => false,
                'message' => 'Order session expired or invalid'
            ], 404);
        }
        foreach ($cachedOrders as $index => $cachedOrder) {
            if (
                !isset(
                    $cachedOrder['order_no'],
                    $cachedOrder['user_type'],
                    $cachedOrder['user_id'],
                    $cachedOrder['cart_item']
                )
            ) {
                \Log::error('Invalid cached order structure', [
                    'index'      => $index,
                    'cacheKey'   => $cacheKey,
                    'cachedData' => $cachedOrder
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Invalid order data. Please try again.'
                ], 422);
            }
        }

        $paymentResponse = [
            'razorpay_callback' => [
                'headers' => $request->headers->all(),
                'body'    => $request->all(),
                'raw'     => $request->getContent(),
            ],
            'processed_at' => now()
        ];

        DB::beginTransaction();

        try {
            $createdOrders = [];
            $orderModels   = [];

            foreach ($cachedOrders as $index => $cachedOrder) {

                $pendingPenalty = ($index === 0) ? ($cachedOrder['pending_penalty'] ?? 0) : 0;


                $order = Order::create([
                    'order_no'            => $this->generateUniqueOrderNo(),
                    'invoice_no'          => $this->generateInvoiceNo(),
                    'user_type'           => $cachedOrder['user_type'],
                    'user_id'             => $cachedOrder['user_id'],
                    'with_shipping'       => $cachedOrder['with_shipping'],
                    'rider_id'            => $cachedOrder['rider_id'],
                    'shipping_address_id' => $cachedOrder['shipping_address_id'],
                    'items_total'         => $cachedOrder['items_total'],
                    'discount_amount'     => $cachedOrder['discount_amount'],
                    'delivery_fee'        => $cachedOrder['delivery_fee'],
                    'grand_total'         => $cachedOrder['grand_total'],
                    'paid_pending'        => $pendingPenalty,   
                    'payment_method'      => 'Razorpay',
                    'payment_status'      => 'paid',
                    'status'              => 'pending_assignment',
                    'paid_at'             => now(),

                    'payment_response'    => $paymentResponse,
                ]);

                $item = $cachedOrder['cart_item'];

                OrderItem::create([
                    'order_id'              => $order->id,
                    'product_id'            => $item['product_id'],
                    'quantity'              => $item['quantity'],
                    'original_unit_price'   => $item['original_unit_price'],
                    'discounted_unit_price' => $item['discounted_unit_price'],
                    'discount_amount'       => $item['discount_amount'],
                    'unit_price'            => $item['unit_price'],
                    'total_price'           => $item['total_price'],
                ]);

                $this->sendOrderPlacedEmail($order);

                $createdOrders[] = $order->id;
                $orderModels[]   = $order;
            }

            if ($cachedOrders[0]['user_type'] === 'store') {
                DB::table('cart_items')->where('store_id', $cachedOrders[0]['user_id'])->delete();
            } else {
                DB::table('cart_items')->where('customer_id', $cachedOrders[0]['user_id'])->delete();
            }

            Cache::forget($cacheKey);

            DB::commit();

            $storeId = $cachedOrders[0]['user_id'] ?? null;
            if ($storeId) {
                DB::table('stores')->where('id', $storeId)->update(['pending_penalty' => 0]);
            }

            foreach ($orderModels as $order) {
                if ($order->with_shipping) {
                    $this->notifyNearbyWarehousesWithProducts($order);
                }
            }

            return response()->json([
                'success'   => true,
                'message'   => 'Order placed successfully',
                'order_ids' => $createdOrders
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Razorpay payment processing error', [
                'message'  => $e->getMessage(),
                'cacheKey' => $cacheKey,
                'trace'    => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to place order'
            ], 500);
        }
    }

    public function createCashfreePaymentLink(Request $request)
    {
        $request->validate(['cache_key' => 'required']);

        $store = auth('store')->user();
        if (!$store) {
            return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
        }

        $cachedOrders = Cache::get($request->cache_key);
        if (!$cachedOrders || !is_array($cachedOrders) || empty($cachedOrders)) {
            return response()->json([
                'success' => false,
                'message' => 'Order session expired'
            ], 410);
        }

        $totalAmount = collect($cachedOrders)->sum('grand_total');
        $pendingPenalty = $cachedOrders[0]['pending_penalty'] ?? 0;
        $totalAmount += $pendingPenalty;
        $firstOrder  = $cachedOrders[0];

        try {
            $response = Http::withHeaders([
                'x-client-id'     => env('CASHFREE_APP_ID'),
                'x-client-secret' => env('CASHFREE_SECRET_KEY'),
                'x-api-version'   => '2022-09-01',
                'Content-Type'    => 'application/json',
            ])->post('https://sandbox.cashfree.com/pg/links', [
                'link_amount'   => (float) $totalAmount,
                'link_currency' => 'INR',
                'link_purpose'  => 'Payment for '.count($cachedOrders).' orders',
                'customer_details' => [
                    'customer_name'  => $firstOrder['customer_name'] ?? $store->name,
                    'customer_phone' => $firstOrder['customer_mobile'] ?? '9999999999',
                ],
                'link_meta' => [
                    'notify_url' => 'https://dewless-marisha-unpenetratingly.ngrok-free.dev/api/store/cashfree/payment-webhook',
                    'return_url' => route('api.cashfree.payment-success'),
                ],
                'link_reference_id' => $request->cache_key,
            ]);

            if (!$response->successful()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cashfree link creation failed',
                    'error'   => $response->body()
                ], 500);
            }

            $data = $response->json();

            $cacheExpiry = now()->addHours(2);
            Cache::put($request->cache_key, $cachedOrders, $cacheExpiry);
            if (isset($data['cf_link_id'])) {
                Cache::put($data['cf_link_id'], $cachedOrders, $cacheExpiry);
            }

            return response()->json([
                'success'      => true,
                'payment_link' => $data['link_url'],
                'cf_link_id'   => $data['cf_link_id'],
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function cashfreeWebhook(Request $request)
    {
        $payload = $request->all();

        \Log::info('Cashfree Webhook HIT', $payload);

        // 🔑 Extract cache key (try cf_link_id, link_id, or original reference)
        $cacheKey = $payload['data']['cf_link_id'] 
            ?? $payload['data']['link_id'] 
            ?? $payload['data']['order']['order_tags']['link_reference_id'] 
            ?? null;

        if (!$cacheKey) {
            \Log::error('Cache key missing in Cashfree webhook');
            return response()->json(['error' => 'Cache key missing'], 400);
        }

        // 🔑 Idempotent check - prevent duplicate processing
        if (Cache::has("processed_$cacheKey")) {
            \Log::info('Cashfree webhook already processed', ['cacheKey' => $cacheKey]);
            return response()->json(['status' => 'already processed'], 200);
        }

        // 🔑 Check payment status
        $paymentStatus = $payload['data']['payment']['payment_status'] 
            ?? $payload['data']['order']['transaction_status'] 
            ?? null;

        if ($paymentStatus !== 'SUCCESS') {
            return response()->json(['status' => 'ignored'], 200);
        }

        // 🔑 Fetch cached orders
        $cachedOrders = Cache::get($cacheKey);
        if (!$cachedOrders || !is_array($cachedOrders) || empty($cachedOrders)) {
            \Log::error('Cached orders not found', ['cacheKey' => $cacheKey]);
            return response()->json(['error' => 'Order session expired'], 404);
        }

        // 🔑 Validate cached order structure
        foreach ($cachedOrders as $index => $cachedOrder) {
            if (!isset($cachedOrder['order_no'], $cachedOrder['user_type'], $cachedOrder['user_id'], $cachedOrder['cart_item'])) {
                \Log::error('Invalid cached order structure', [
                    'index' => $index,
                    'cacheKey' => $cacheKey,
                    'cachedData' => $cachedOrder
                ]);
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid order data. Please try again.'
                ], 422);
            }
        }

        // 🔑 Prepare full Cashfree transaction JSON
        $paymentResponse = [
            'cashfree_callback' => [
                'headers' => $request->headers->all(),
                'body'    => $payload,
                'raw'     => $request->getContent(),
            ],
            'processed_at' => now(),
        ];

        DB::beginTransaction();

        try {
            $createdOrders = [];
            $orderModels   = [];

            foreach ($cachedOrders as $index => $cachedOrder) {
            
            $paidPending = ($index === 0) ? ($cachedOrder['pending_penalty'] ?? 0) : 0;
                $order = Order::create([
                    'order_no'            => $this->generateUniqueOrderNo(),
                    'invoice_no'          => $this->generateInvoiceNo(),
                    'user_type'           => $cachedOrder['user_type'],
                    'user_id'             => $cachedOrder['user_id'],
                    'with_shipping'       => $cachedOrder['with_shipping'],
                    'rider_id'            => $cachedOrder['rider_id'],
                    'shipping_address_id' => $cachedOrder['shipping_address_id'],
                    'items_total'         => $cachedOrder['items_total'],
                    'discount_amount'     => $cachedOrder['discount_amount'],
                    'delivery_fee'        => $cachedOrder['delivery_fee'],
                    'grand_total'         => $cachedOrder['grand_total'],
                    'paid_pending'        => $paidPending,
                    'payment_method'      => 'Cashfree',
                    'payment_status'      => 'paid',
                    'status'              => 'pending_assignment',
                    'paid_at'             => now(),
                    'payment_response'    => $paymentResponse,
                ]);

                // ✅ Create order items
                $item = $cachedOrder['cart_item'];
                OrderItem::create([
                    'order_id'              => $order->id,
                    'product_id'            => $item['product_id'],
                    'quantity'              => $item['quantity'],
                    'original_unit_price'   => $item['original_unit_price'],
                    'discounted_unit_price' => $item['discounted_unit_price'],
                    'discount_amount'       => $item['discount_amount'],
                    'unit_price'            => $item['unit_price'],
                    'total_price'           => $item['total_price'],
                ]);

                $this->sendOrderPlacedEmail($order);

                $createdOrders[] = $order->id;
                $orderModels[]   = $order;
            }

            if ($cachedOrders[0]['user_type'] === 'store') {
                DB::table('cart_items')->where('store_id', $cachedOrders[0]['user_id'])->delete();
            } else {
                DB::table('cart_items')->where('customer_id', $cachedOrders[0]['user_id'])->delete();
            }

            $storeId = $cachedOrders[0]['user_id'];
            DB::table('stores')->where('id', $storeId)->update(['pending_penalty' => 0]);

            Cache::put("processed_$cacheKey", true, now()->addMinutes(30));
            Cache::forget($cacheKey);

            DB::commit();

            foreach ($orderModels as $order) {
                if ($order->with_shipping) {
                    $this->notifyNearbyWarehousesWithProducts($order);
                }
            }

            return response()->json([
                'success'   => true,
                'message'   => 'Order placed successfully',
                'order_ids' => $createdOrders
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Cashfree payment processing error', [
                'message'  => $e->getMessage(),
                'cacheKey' => $cacheKey,
                'trace'    => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to place order'
            ], 500);
        }
    }

    public function cashfreePaymentSuccess(Request $request)
    {
        return response()->json([
            'success' => true,
            'message' => 'Payment successful. Awaiting confirmation...'
        ]);
    }

    private function notifyNearbyWarehousesWithProducts($order, $radiusKm = 10)
    {
        if (!$order) return;

        if ($order instanceof \stdClass) {
            $order = Order::find($order->id);
            if (!$order) return;
        }

        if (!$order->shipping_address_id) return;

        $address = DB::table('addresses')->where('id', $order->shipping_address_id)->first();
        if (!$address || !$address->latitude || !$address->longitude) return;

        $lat = $address->latitude;
        $lng = $address->longitude;

        $warehouses = DB::table('warehouses')
            ->select(
                'id',
                'warehouse_name',
                'latitude',
                'longitude',
                DB::raw("(
                    6371 * acos(
                        cos(radians($lat)) *
                        cos(radians(latitude)) *
                        cos(radians(longitude) - radians($lng)) +
                        sin(radians($lat)) *
                        sin(radians(latitude))
                    )
                ) AS distance")
            )
            ->whereNotNull('latitude')
            ->whereNotNull('longitude')
            ->having('distance', '<=', $radiusKm)
            ->get();

        if ($warehouses->isEmpty()) return;

        // No need to check products for notification, just send for each order
        foreach ($warehouses as $warehouse) {
            $this->sendNotification(
                'New Order Available',
                'A new order is available near your location. Tap to view nearby orders.',
                'warehouse',
                $warehouse->id,
                'warehouse/nearby-orders',
                $order->id // optional: link notification to this order
            );
        }
    }

    private function sendNotification($title, $content, $userType, $userId, $route = null, $routeId = null)
    {
        $linkData = $route ? ($routeId ? json_encode(['route'=>$route,'route_id'=>$routeId]) : json_encode(['route'=>$route])) : null;

        DB::table('notifications')->insert([
            'title' => $title,
            'content' => $content,
            'user_type' => $userType,
            'user_id' => $userId,
            'status' => 'active',
            'link' => $linkData,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        $tokens = DB::table('fcm_tokens')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('notify_status', 'active')
            ->pluck('fcm_token')
            ->toArray();

        if (empty($tokens)) return;

        $credentialsPath = base_path(env('FIREBASE_CREDENTIALS'));
        $projectId = env('FIREBASE_PROJECT_ID');

        $client = new \Google_Client();
        $client->setAuthConfig($credentialsPath);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');
        $bearerToken = $client->fetchAccessTokenWithAssertion()['access_token'] ?? null;
        if (!$bearerToken) return;

        $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

        foreach ($tokens as $fcmToken) {
            $payload = [
                "message" => [
                    "token" => $fcmToken,
                    "notification" => ["title" => $title, "body" => $content],
                    "data" => ["route" => (string)$route, "type"=>"notification","user_type"=>$userType,"status"=>"active"] + ($routeId ? ["route_id" => (string)$routeId] : []),
                    "android" => ["priority"=>"high","notification"=>["sound"=>"default","click_action"=>"FLUTTER_NOTIFICATION_CLICK"]]
                ]
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ["Authorization: Bearer {$bearerToken}", "Content-Type: application/json"]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
            $response = curl_exec($ch);
            if ($response === false) \Log::error('Firebase Push Error: '.curl_error($ch));
            curl_close($ch);
        }
    }

    public function getMyStoreOrders(Request $request)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 401);
        }

        $filterMap = [
            'ordered' => [
                'pending_assignment',
                'pending_payment',
                'order_taken',
                'waiting_for_rider_approval',
                'rider_accepted',
                'rider_verified',
            ],
            'loaded' => [
                'in_transit',
            ],
            'delivered' => [
                'completed',
            ],
            'cancelled' => [
                'cancelled',
            ],
        ];

        $filter = $request->query('filter');

        $ordersQuery = DB::table('orders as o')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->where('o.user_type', 'store')
            ->where('o.user_id', $store->id);

        if ($filter && isset($filterMap[$filter])) {
            $ordersQuery->whereIn('o.status', $filterMap[$filter]);
        }

        $orders = $ordersQuery
            ->orderByDesc('o.created_at')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.created_at',
                'o.cancelled_at',
                'o.paid_at',
                'o.payment_method',
                'o.rider_id',
                'r.rider_code'
            )
            ->get();

        $orders = $orders->map(function ($order) {

            $baseDate = $order->paid_at ?? $order->created_at;
            $baseDate = Carbon::parse($baseDate);

            $displayStatus   = '';
            $deliveryMessage = '';
            $deliveryTime    = null;

            if ($order->status === 'pending_assignment') {
                $displayStatus   = 'Order is being processed';
                $deliveryMessage = 'Assigning delivery partner';

            } elseif (in_array($order->status, [
                'order_taken',
                'waiting_for_rider_approval',
                'rider_accepted',
                'rider_verified'
            ])) {
                $displayStatus = 'Arriving soon';
                $deliveryDate = $baseDate->copy()->addDays(7);
                $deliveryMessage = 'Delivery expected by ' . $deliveryDate->format('D, d M');

            } elseif ($order->status === 'in_transit') {
                $displayStatus   = 'Arriving today';
                $deliveryMessage = 'Delivery expected today';
                $deliveryTime    = '9:00 PM';

            } elseif ($order->status === 'completed') {
                $displayStatus   = 'Delivered';
                $deliveryMessage = 'Order delivered successfully';

            } elseif ($order->status === 'cancelled') {
                $displayStatus = 'Cancelled';

                if ($order->cancelled_at) {
                    $cancelledDate = Carbon::parse($order->cancelled_at)->format('d M Y');
                    $deliveryMessage = 'As per your request on, ' . $cancelledDate;
                } else {
                    $deliveryMessage = 'Order cancelled';
                }
            } else {
                $displayStatus   = ucfirst(str_replace('_', ' ', $order->status));
                $deliveryMessage = '';
            }

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'oi.quantity',
                    'oi.status as order_item_status',
                    'p.colors',
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category'    => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand'       => $item->brand,
                        'quantity'    => $item->quantity,
                        'unit'        => $item->quantity_unit,
                        'color'       => $item->colors ? json_decode($item->colors) : null,
                        'status'      => $item->order_item_status,
                        'image'       => $item->image ? asset($item->image) : null,
                    ];
                });

            return [
                'order_id' => $order->id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'display_status'   => $displayStatus,
                'delivery_message' => $deliveryMessage,
                'delivery_time'    => $deliveryTime,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'date' => $baseDate->format('Y-m-d'),
                'time' => $baseDate->format('H:i:s'),
                'order_items' => $items,
            ];
        });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'filter'  => $filter ?? 'all',
            'data'    => $orders,
        ]);
    }

    public function cancelOrder(Request $request, $order_id)
    {
        $request->validate([
            'reason_type' => 'required|string',
            'remark'      => 'nullable|string|max:255'
        ]);

        $store = auth('store')->user();

        $order = Order::where('id', $order_id)
            ->where('user_id', $store->id)
            ->where('user_type', 'store')
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $blockedStatuses = ['rider_verified', 'in_transit', 'completed'];

        if (in_array($order->status, $blockedStatuses)) {
            return response()->json([
                'success' => false,
                'message' => 'Order cannot be cancelled at this stage'
            ], 400);
        }

        if (now()->diffInMinutes($order->created_at) > 10) {
            return response()->json([
                'success' => false,
                'message' => 'Order can only be cancelled within 10 minutes'
            ], 400);
        }

        $cancelReason = [
            'reason_type' => $request->reason_type,
            'remark'      => $request->remark
        ];

        $order->status = 'cancelled';
        $order->seller_payment_status = 'cancelled';
        $order->cancellation_charge = 200;
        $order->cancel_reason = json_encode($cancelReason);
        $order->cancelled_at = now();  
        $order->save();

        OrderItem::where('order_id', $order->id)
            ->update(['status' => 'cancelled']);

        OrderRiderRequest::where('order_id', $order->id)
            ->update(['status' => 'expired']);

        if (in_array($order->payment_method, ['Razorpay', 'Cashfree'])) {

            $order->payment_status = 'refunded';
            $order->save();

            return response()->json([
                'success' => true,
                'message' => 'Order cancelled. ₹200 cancellation charge recorded. Refund will be processed separately.'
            ]);
        }

        if ($order->payment_method === 'COD') {

            $penaltyAmount = 200 + ($order->paid_pending ?? 0);

            if ($order->user_type === 'customer') {
                Customer::where('id', $order->user_id)
                    ->increment('pending_penalty', $penaltyAmount);
            }

            if ($order->user_type === 'store') {
                Store::where('id', $order->user_id)
                    ->increment('pending_penalty', $penaltyAmount);
            }

            return response()->json([
                'success' => true,
                'message' => 'Order cancelled. ₹' . $penaltyAmount . ' penalty added to your account.'
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Order cancelled successfully'
        ]);
    }

    public function getStoreOrderDetail(Request $request, $orderId)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $order = DB::table('orders as o')
            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as d', 'd.id', '=', 'da.district_id')
            ->leftJoin('states as s', 's.id', '=', 'da.state_id')
            ->where('o.id', $orderId)
            ->where('o.user_type', 'store')
            ->where('o.user_id', $store->id)
            ->select(
                'o.*',
                'da.name as drop_name',
                'da.mobile_number as drop_phone',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'd.name as drop_district',
                's.name as drop_state',
                'da.pincode as drop_pincode',
                'da.latitude as drop_lat',
                'da.longitude as drop_lng'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $orderDate = ($order->payment_method === 'COD')
            ? $order->created_at
            : ($order->paid_at ?? $order->created_at);

        $items = DB::table('order_items as oi')
            ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $order->id)
            ->select(
                'oi.product_id',
                'oi.quantity',
                'oi.status as order_item_status',
                'p.colors',
                'p.image',
                'c.name as category',
                's.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit'
            )
            ->get()
            ->map(function ($item) use ($order) {
                $review = null;

                if ($order->status === 'completed') {
                    $reviewData = DB::table('product_reviews')
                        ->where('order_id', $order->id)
                        ->where('product_id', $item->product_id)
                        ->first();

                    if ($reviewData) {
                        $review = [
                            'rating' => $reviewData->rating,
                            'comment'=> $reviewData->review
                        ];
                    }
                }
                return [
                    'product_id' => $item->product_id,
                    'category'   => $item->category,
                    'subcategory'=> $item->subcategory,
                    'brand'      => $item->brand,
                    'piece'      => $item->quantity,
                    'unit'       => $item->quantity_unit,
                    'color'      => $item->colors ? json_decode($item->colors) : null,
                    'image'      => $item->image ? asset($item->image) : null,
                    'review'     => $review
                ];
            });
        $timeline = [];

        function formatDateTime($dt) {
            if (!$dt) return ['date' => null, 'time' => null];

            $c = Carbon::parse($dt);
            return [
                'date' => $c->format('d-m-Y'),   // 21-01-2026
                'time' => $c->format('h:i A'),   // 09:10 AM
            ];
        }

        if (in_array($order->status, [
            'pending_assignment',
            'order_taken',
            'waiting_for_rider_approval',
            'rider_accepted',
            'rider_verified',
            'in_transit',
            'completed'
        ])) {

            $dt = formatDateTime($orderDate);

            $timeline[] = [
                'title'     => 'Order Confirmed',
                'message'   => 'Your order has been successfully confirmed.',
                'date'      => $dt['date'],
                'time'      => $dt['time'],
                'completed' => true
            ];
        }

        if (!empty($order->ride_started_at)) {

            $dt = formatDateTime($order->ride_started_at);

            $timeline[] = [
                'title'     => 'Material Loaded',
                'message'   => 'Material loaded and ready for delivery.',
                'date'      => $dt['date'],
                'time'      => $dt['time'],
                'completed' => true
            ];

        } else {

            $timeline[] = [
                'title'     => 'Material Loaded',
                'message'   => 'Material loading pending.',
                'date'      => null,
                'time'      => null,
                'completed' => false
            ];
        }

        if (!empty($order->delivered_at)) {

            $dt = formatDateTime($order->delivered_at);

            $timeline[] = [
                'title'     => 'Delivered',
                'message'   => 'Material has been successfully delivered to the site.',
                'date'      => $dt['date'],
                'time'      => $dt['time'],
                'completed' => true
            ];

        } else {

            $timeline[] = [
                'title'     => 'Delivered',
                'message'   => 'Delivery in progress.',
                'date'      => null,
                'time'      => null,
                'completed' => false
            ];
        }

        $dropLocation = [
            'name'      => $order->drop_name,
            'phone'     => $order->drop_phone,
            'door_no'   => $order->drop_door_no,
            'street'    => $order->drop_street,
            'area'      => $order->drop_area,
            'city'      => $order->drop_city,
            'district'  => $order->drop_district,
            'state'     => $order->drop_state,
            'pincode'   => $order->drop_pincode,
        ];

        /* ---------------- DELIVERY OTP LOGIC ---------------- */
        $deliveryOtp = null;

        if ($order->status === 'in_transit') {
            $deliveryOtp = $order->delivery_otp;
        }

        /* ---------------- BILLING DETAILS ---------------- */

        $totalMaterialPrice = (float) $order->items_total;

        $savedAmount = (float) ($order->discount_amount ?? 0);

        $deliveryCharge = $order->with_shipping ? (float) $order->delivery_fee : 0;

        $totalPayable = (float) $order->grand_total;

        $paymentLabel = null;

        if ($order->payment_method === 'COD') {
            $paymentLabel = 'Cash on Delivery';
        } elseif ($order->payment_method === 'Razorpay') {
            $paymentLabel = 'Prepaid (Razorpay)';
        } elseif ($order->payment_method === 'Cashfree') {
            $paymentLabel = 'Prepaid (Cashfree)';
        }

        $orderCreatedAt = Carbon::parse($order->created_at);
        $cancelAllowedUntil = $orderCreatedAt->copy()->addMinutes(10);
        $now = Carbon::now();

        $canCancel = $now->lt($cancelAllowedUntil);

        $remainingSeconds = $canCancel
            ? $now->diffInSeconds($cancelAllowedUntil)
            : 0;

        return response()->json([
            'success' => true,
            'data' => [
                'order_id'   => $order->id,
                'order_no'   => $order->order_no,
                'invoice_no' => $order->invoice_no,

                'order_date' => date('Y-m-d', strtotime($orderDate)),
                'order_time' => date('H:i:s', strtotime($orderDate)),

                'status'     => $order->status,
                'items'         => $items,
                'delivery_otp' => ($order->status === 'in_transit') ? $order->delivery_otp : null,
                'tracking_timeline' => $timeline,
                'drop_location' => $dropLocation,
                'billing' => [
                    'total_material_price' => round($totalMaterialPrice, 2),
                    'saved_amount'         => round($savedAmount, 2),
                    'delivery_charge'      => round($deliveryCharge, 2),
                    'total_payable'        => round($totalPayable, 2),
                    'payment_method'       => $paymentLabel,
                ],
                'cancel_info' => [
                    'can_cancel' => $canCancel,
                    'remaining_seconds' => $remainingSeconds,
                    'cancel_allowed_until' => $cancelAllowedUntil->format('Y-m-d H:i:s')
                ],

            ]
        ]);
    }

    public function submitProductReview(Request $request)
    {
        $customer = auth('customer')->user();
        $store    = auth('store')->user();

        if (!$customer && !$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'order_id'   => 'required|exists:orders,id',
            'product_id' => 'required|exists:products,id',
            'rating'     => 'required|integer|min:1|max:5',
            'review'     => 'nullable|string|max:1000'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors'  => $validator->errors()
            ], 422);
        }

        $userType = $customer ? 'customer' : 'store';
        $userId   = $customer ? $customer->id : $store->id;

        $order = DB::table('orders')
            ->where('id', $request->order_id)
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid order for this user'
            ], 403);
        }

        if ($order->status !== 'completed') {
            return response()->json([
                'success' => false,
                'message' => 'Review allowed only after order is delivered'
            ], 403);
        }

        $orderItem = DB::table('order_items')
            ->where('order_id', $request->order_id)
            ->where('product_id', $request->product_id)
            ->first();

        if (!$orderItem) {
            return response()->json([
                'success' => false,
                'message' => 'This product is not part of this order'
            ], 403);
        }

        $alreadyReviewed = DB::table('product_reviews')
            ->where('order_id', $request->order_id)
            ->where('product_id', $request->product_id)
            ->where($userType.'_id', $userId)
            ->first();

        if ($alreadyReviewed) {
            return response()->json([
                'success' => false,
                'message' => 'You already reviewed this product for this order'
            ], 409);
        }

        DB::table('product_reviews')->insert([
            'order_id'    => $request->order_id,
            'product_id'  => $request->product_id,
            'customer_id' => $customer ? $customer->id : null,
            'store_id'    => $store ? $store->id : null,
            'rating'      => $request->rating,
            'review'      => $request->review,
            'created_at'  => now(),
            'updated_at'  => now()
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Review submitted successfully'
        ]);
    }

    // private function getInvoiceData($orderId)
    // {
    //     $store = auth('store')->user();
    //     if (!$store) {
    //         return null;
    //     }

    //     $order = DB::table('orders as o')
    //         ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
    //         ->leftJoin('districts as d', 'd.id', '=', 'da.district_id')
    //         ->leftJoin('states as s', 's.id', '=', 'da.state_id')
    //         ->where('o.id', $orderId)
    //         ->where('o.user_type', 'store')
    //         ->where('o.user_id', $store->id)
    //         ->select(
    //             'o.*',
    //             'da.name as drop_name',
    //             'da.mobile_number as drop_phone',
    //             'da.door_no as drop_door_no',
    //             'da.street_name as drop_street',
    //             'da.area as drop_area',
    //             'da.city as drop_city',
    //             'd.name as drop_district',
    //             's.name as drop_state',
    //             'da.pincode as drop_pincode'
    //         )
    //         ->first();

    //     if (!$order) {
    //         return null;
    //     }

    //     $invoiceDateTime = $order->paid_at ?? $order->created_at;

    //     $invoiceDate = Carbon::parse($invoiceDateTime)->format('d-m-Y');
    //     $invoiceTime = Carbon::parse($invoiceDateTime)->format('h:i A');

    //     $orderDate = ($order->payment_method === 'COD')
    //         ? $order->created_at
    //         : ($order->paid_at ?? $order->created_at);

    //     /* ---------- MATERIALS ---------- */

    //     $materials = DB::table('order_items as oi')
    //         ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
    //         ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
    //         ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
    //         ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
    //         ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
    //         ->where('oi.order_id', $order->id)
    //         ->select(
    //             'oi.product_id',
    //             'oi.quantity',
    //             'oi.unit_price',
    //             DB::raw('(oi.quantity * oi.unit_price) as total_price'),
    //             'p.image',
    //             'c.name as category',
    //             'sc.name as subcategory',
    //             'b.name as brand',
    //             'q.name as quantity_unit'
    //         )
    //         ->get()
    //         ->map(function ($item) use ($order) {
    //             return [
    //                 'product_id'    => $item->product_id,
    //                 'category'      => $item->category,
    //                 'subcategory'   => $item->subcategory,
    //                 'brand'         => $item->brand,
    //                 'quantity'      => $item->quantity,
    //                 'quantity_unit' => $item->quantity_unit,
    //                 'unit_price'    => round($item->unit_price, 2),
    //                 'total_price'   => round($item->total_price, 2),
    //                 'image'         => $item->image ?? null,
    //             ];
    //         });


    //     /* ---------- DELIVERY ADDRESS ---------- */

    //     $deliveryAddress = [
    //         'name'     => $order->drop_name,
    //         'phone'    => $order->drop_phone,
    //         'door_no'  => $order->drop_door_no,
    //         'street'   => $order->drop_street,
    //         'area'     => $order->drop_area,
    //         'city'     => $order->drop_city,
    //         'district' => $order->drop_district,
    //         'state'    => $order->drop_state,
    //         'pincode'  => $order->drop_pincode,
    //     ];

    //     /* ---------- PAYMENT SUMMARY ---------- */

    //     $paymentMethodLabel = match ($order->payment_method) {
    //         'COD'       => 'Cash on Delivery',
    //         'Razorpay'  => 'Prepaid (Razorpay)',
    //         'Cashfree'  => 'Prepaid (Cashfree)',
    //          default     => 'Unknown',
    //     };

    //     $transactionId = in_array($order->payment_method, ['Razorpay', 'Cashfree'])
    //         ? $order->gateway_payment_id
    //         : null;

    //     return [
    //         'invoice_no'   => $order->invoice_no,
    //         'invoice_date' => $invoiceDate,
    //         'invoice_time' => $invoiceTime,
    //         'order_no'     => $order->order_no,
    //         'order_date'   => Carbon::parse($orderDate)->format('d-m-Y'),
    //         'order_id'     => $order->id,
    //         'materials'    => $materials,
    //         'delivery_address' => $deliveryAddress,
    //         'payment_summary' => [
    //             'grand_total'      => round($order->grand_total, 2),
    //             'delivery_charge' => $order->with_shipping
    //                 ? round($order->delivery_fee, 2)
    //                 : 0,
    //             'payment_method'  => $paymentMethodLabel,
    //             'payment_status'  => ucfirst($order->payment_status),
    //             'transaction_id'  => $transactionId
    //         ]
    //     ];
    // }
    private function getInvoiceData($orderId)
    {
        // $store = auth('store')->user();
        // if (!$store) {
        //     return null;
        // }

        $order = DB::table('orders as o')
            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as d', 'd.id', '=', 'da.district_id')
            ->leftJoin('states as s', 's.id', '=', 'da.state_id')
            ->where('o.id', $orderId)
            ->where('o.user_type', 'store')
            ->select(
                'o.*',
                'da.name as drop_name',
                'da.mobile_number as drop_phone',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'd.name as drop_district',
                's.name as drop_state',
                'da.pincode as drop_pincode'
            )
            ->first();


        if (!$order) {
            return null;
        }

        $invoiceDateTime = $order->paid_at ?? $order->created_at;

        $invoiceDate = Carbon::parse($invoiceDateTime)->format('d-m-Y');
        $invoiceTime = Carbon::parse($invoiceDateTime)->format('h:i A');

        $orderDate = ($order->payment_method === 'COD')
            ? $order->created_at
            : ($order->paid_at ?? $order->created_at);

        /* ---------- MATERIALS ---------- */

        $materials = DB::table('order_items as oi')
            ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $order->id)
            ->select(
                'oi.product_id',
                'oi.quantity',
                'oi.unit_price',
                DB::raw('(oi.quantity * oi.unit_price) as total_price'),
                'p.image',
                'c.name as category',
                'sc.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit'
            )
            ->get()
            ->map(function ($item) use ($order) {
                return [
                    'product_id'    => $item->product_id,
                    'category'      => $item->category,
                    'subcategory'   => $item->subcategory,
                    'brand'         => $item->brand,
                    'quantity'      => $item->quantity,
                    'quantity_unit' => $item->quantity_unit,
                    'unit_price'    => round($item->unit_price, 2),
                    'total_price'   => round($item->total_price, 2),
                    'image'         => $item->image ?? null,
                ];
            });


        /* ---------- DELIVERY ADDRESS ---------- */

        $deliveryAddress = [
            'name'     => $order->drop_name,
            'phone'    => $order->drop_phone,
            'door_no'  => $order->drop_door_no,
            'street'   => $order->drop_street,
            'area'     => $order->drop_area,
            'city'     => $order->drop_city,
            'district' => $order->drop_district,
            'state'    => $order->drop_state,
            'pincode'  => $order->drop_pincode,
        ];

        /* ---------- PAYMENT SUMMARY ---------- */

        $paymentMethodLabel = match ($order->payment_method) {
            'COD'       => 'Cash on Delivery',
            'Razorpay'  => 'Prepaid (Razorpay)',
            'Cashfree'  => 'Prepaid (Cashfree)',
            default     => 'Unknown',
        };

        $transactionId = in_array($order->payment_method, ['Razorpay', 'Cashfree'])
            ? $order->gateway_payment_id
            : null;

        return [
            'invoice_no'   => $order->invoice_no,
            'invoice_date' => $invoiceDate,
            'invoice_time' => $invoiceTime,
            'order_no'     => $order->order_no,
            'order_date'   => Carbon::parse($orderDate)->format('d-m-Y'),
            'order_id'     => $order->id,
            'materials'    => $materials,
            'delivery_address' => $deliveryAddress,
            'payment_summary' => [
                'grand_total'      => round($order->grand_total, 2),
                'delivery_charge' => $order->with_shipping
                    ? round($order->delivery_fee, 2)
                    : 0,
                'payment_method'  => $paymentMethodLabel,
                'payment_status'  => ucfirst($order->payment_status),
                'transaction_id'  => $transactionId
            ]
        ];
    }

    public function getStoreOrderInvoice(Request $request, $orderId)
    {
        $invoice = $this->getInvoiceData($orderId);

        if (!$invoice) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'invoice' => $invoice
        ]);
    }

    public function downloadStoreOrderInvoicePdf($orderId)
    {
        $order = DB::table('orders')->where('id', $orderId)->first();

        $paymentResponse = json_decode($order->payment_response, true);

        // Extract Razorpay transaction details if available
        $transactionDetails = [];
        if (isset($paymentResponse['razorpay_callback']['body'])) {
            $body = $paymentResponse['razorpay_callback']['body'];
            $transactionDetails = [
                'payment_id' => $body['razorpay_payment_id'] ?? null,
                // 'payment_link_id' => $body['razorpay_payment_link_id'] ?? null,
                // 'reference_id' => $body['razorpay_payment_link_reference_id'] ?? null,
                'status' => $body['razorpay_payment_link_status'] ?? null,
                // 'signature' => $body['razorpay_signature'] ?? null,
            ];
        }

        // $invoiceData = [
        //     'invoice_no' => $order->invoice_no,
        //     'order_no' => $order->order_no,
        //     'order_date' => $order->created_at,
        //     'grand_total' => $order->grand_total,
        //     'paid_pending' => $order->paid_pending,
        //     'payment_status' => $order->payment_status,
        //     'payment_method' => $order->payment_method,
        //     'transaction_details' => $transactionDetails,
        //     // include materials and delivery address here...
        // ];
        $invoice = $this->getInvoiceData($orderId);

        if (!$invoice) {
            abort(404, 'Invoice not found');
        }

        $pdf = Pdf::loadView('pdf.store-purchase', [
        'invoice' => $invoice + [
            'paid_pending' => $order->paid_pending,
            'transaction_details' => $transactionDetails,
        ]
        ])->setPaper('a4', 'portrait');

        $fileName = 'Invoice-' . $invoice['invoice_no'] . '.pdf';

        return $pdf->download($fileName);
    }

    public function retailPurchaseReport(Request $request)
    {
        $retailUser = auth('store')->user();

        $startDate = null;
        $endDate   = null;

        if ($request->filled('date')) {
            $startDate = Carbon::parse($request->date)->startOfDay();
            $endDate   = Carbon::parse($request->date)->endOfDay();
        } elseif ($request->filled('from_date') && $request->filled('to_date')) {
            $startDate = Carbon::parse($request->from_date)->startOfDay();
            $endDate   = Carbon::parse($request->to_date)->endOfDay();
        } elseif ($request->filled('month') && $request->filled('year')) {
            $startDate = Carbon::create($request->year, $request->month, 1)->startOfMonth();
            $endDate   = Carbon::create($request->year, $request->month, 1)->endOfMonth();
        } elseif ($request->filled('year')) {
            $startDate = Carbon::create($request->year, 1, 1)->startOfYear();
            $endDate   = Carbon::create($request->year, 12, 31)->endOfYear();
        } else {
            // Default: current month
            $startDate = Carbon::now()->startOfMonth();
            $endDate   = Carbon::now();
        }

        $allOrdersQuery = DB::table('orders')
            ->where('user_type', 'store')
            ->where('user_id', $retailUser->id);

        $completedOrdersQuery = (clone $allOrdersQuery)
            ->where('status', 'completed')
            ->whereBetween('delivered_at', [$startDate, $endDate]);

        $summary = (clone $completedOrdersQuery)
            ->select(
                DB::raw('SUM(grand_total) as total_purchase_amount'),
                DB::raw('COUNT(*) as received_orders'),
                DB::raw('SUM(CASE WHEN with_shipping = 1 THEN 1 ELSE 0 END) as with_shipping_orders'),
                DB::raw('SUM(CASE WHEN with_shipping = 0 THEN 1 ELSE 0 END) as without_shipping_orders'),
                DB::raw('(SELECT SUM(quantity) FROM order_items oi
                        JOIN orders o2 ON o2.id = oi.order_id
                        WHERE o2.user_type = "store" AND o2.user_id = '.$retailUser->id.'
                        AND o2.status = "completed"
                        AND o2.delivered_at BETWEEN "'.$startDate.'" AND "'.$endDate.'") as materials_received')
            )
            ->first();

        $totalOrders = (clone $allOrdersQuery)->count(); // all orders count

        $categorySummary = DB::table('order_items as oi')
            ->join('orders as o', 'o.id', '=', 'oi.order_id')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->join('categories as c', 'c.id', '=', 'p.category_id')
            ->where('o.user_type', 'store')
            ->where('o.user_id', $retailUser->id)
            ->where('o.status', 'completed')
            ->when($request->filled('date'), fn($q) => $q->whereDate('o.delivered_at', $request->date))
            ->when($request->filled('from_date') && $request->filled('to_date'), 
                fn($q) => $q->whereBetween('o.delivered_at', [$request->from_date . ' 00:00:00', $request->to_date . ' 23:59:59']))
            ->when($request->filled('month') && $request->filled('year'), 
                fn($q) => $q->whereMonth('o.delivered_at', $request->month)->whereYear('o.delivered_at', $request->year))
            ->when($request->filled('year') && !$request->filled('month'), 
                fn($q) => $q->whereYear('o.delivered_at', $request->year))
            ->select(
                'c.id as category_id',
                'c.name as category_name',
                'c.image as category_image',
                'p.quantity_unit as unit',
                DB::raw('SUM(oi.quantity) as total_quantity')
            )
            ->groupBy('c.id', 'p.quantity_unit')
            ->orderBy('c.name')
            ->get()
            ->map(function ($item) {
                return [
                    'category_id'    => $item->category_id,
                    'category_name'  => $item->category_name,
                    'category_image' => $item->category_image ? asset($item->category_image) : null,
                    'unit'           => $item->unit,
                    'total_quantity' => $item->total_quantity
                ];
            });

        $paymentSummary = (clone $completedOrdersQuery)
            ->select(
                DB::raw("SUM(CASE WHEN payment_method = 'COD' THEN grand_total ELSE 0 END) as cod_amount"),
                DB::raw("SUM(CASE WHEN payment_method IN ('Razorpay','Cashfree') THEN grand_total ELSE 0 END) as online_amount")
            )
            ->first();

        return response()->json([
            'success' => true,
            'filters' => [
                'start_date' => $startDate->format('Y-m-d'),
                'end_date'   => $endDate->format('Y-m-d'),
            ],
            'summary' => [
                'total_purchase_amount'   => (float) ($summary->total_purchase_amount ?? 0),
                'total_orders'            => (int)   ($totalOrders ?? 0),
                'received_orders'         => (int)   ($summary->received_orders ?? 0),
                'materials_received'      => (float) ($summary->materials_received ?? 0),
                'with_shipping_orders'    => (int)   ($summary->with_shipping_orders ?? 0),
                'without_shipping_orders' => (int)   ($summary->without_shipping_orders ?? 0),
            ],
            'category_summary' => $categorySummary,
            'payment_summary' => [
                'cod_amount'    => (float) ($paymentSummary->cod_amount ?? 0),
                'online_amount' => (float) ($paymentSummary->online_amount ?? 0),
            ]
        ]);
    }

    public function storePurchaseDashboard(Request $request)
    {
        $store = auth('store')->user();

        if (!$store) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $today = Carbon::today();
        $lastWeekStart = Carbon::now()->subWeek()->startOfWeek(Carbon::MONDAY);
        $lastWeekEnd   = Carbon::now()->subWeek()->endOfWeek(Carbon::SUNDAY);

        $totalPurchaseOrders = DB::table('orders')
            ->where('user_type', 'store')
            ->where('user_id', $store->id)
            ->where('status', 'completed')
            ->count();

        $topCategories = DB::table('order_items as oi')
            ->join('orders as o', 'o.id', '=', 'oi.order_id')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->join('categories as c', 'c.id', '=', 'p.category_id')
            ->where('o.user_type', 'store')
            ->where('o.user_id', $store->id)
            ->where('o.status', 'completed')
            ->select(
                'c.id as category_id',
                'c.name as category_name',
                DB::raw('SUM(oi.quantity) as total_quantity')
            )
            ->groupBy('c.id')
            ->orderByDesc('total_quantity')
            ->limit(3)
            ->get();

        $lastWeekMaterials = DB::table('order_items as oi')
            ->join('orders as o', 'o.id', '=', 'oi.order_id')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->join('categories as c', 'c.id', '=', 'p.category_id')
            ->where('o.user_type', 'store')
            ->where('o.user_id', $store->id)
            ->where('o.status', 'completed')
            ->whereBetween('o.approved_at', [$lastWeekStart, $lastWeekEnd])
            ->select(
                'c.id as category_id',
                'c.name as category_name',
                DB::raw('COUNT(DISTINCT o.id) as orders_count')
            )
            ->groupBy('c.id')
            ->orderByDesc('orders_count')
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'total_purchase_orders' => $totalPurchaseOrders,
                'top_categories'        => $topCategories,
                'last_week_materials'   => $lastWeekMaterials
            ]
        ]);
    }


}