<?php

namespace App\Http\Controllers;

use App\Models\Address;
use App\Models\District;
use App\Models\Product;
use App\Models\QuantityUnit;
use App\Models\State;
use App\Models\WarehouseProduct;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Razorpay\Api\Api as RazorpayApi;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Models\Warehouse;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;
use Razorpay\Api\Errors\SignatureVerificationError;
use Tymon\JWTAuth\Facades\JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;
use Kreait\Firebase\Factory;
use Razorpay\Api\Api;


class WarehouseController extends Controller
{
    // ================= REGISTER =================
    public function initiateRegistration(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'owner_name' => 'required|string|max:255',
            'email' => 'required|email|unique:warehouses,email',
            'mobile_number' => 'required|unique:warehouses,mobile_number',
            'password' => 'required|min:8',
            'warehouse_name' => 'required|string|max:255',
            'door_no' => 'required',
            'street_name' => 'required',
            'area' => 'required',
            'city' => 'required',
            'state_id' => 'required|exists:states,id',
            'district_id' => 'required|exists:districts,id',
            'pincode' => 'required',
            'country' => 'required',
            'lat' => 'required|numeric',
            'lon' => 'required|numeric',
            'pan_number' => 'required|unique:warehouses,pan_number',
            'aadhar_number' => 'required|digits:12|unique:warehouses,aadhar_number',
            'business_type' => 'nullable|in:Individual,Proprietorship,Partnership,Private Limited',
            'business_status' => 'required|in:registered,unregistered',
            'gst_number' => 'nullable|required_if:business_status,registered|unique:warehouses,gst_number',
            'trade_license_number' => 'nullable|required_if:business_status,registered',
            'products' => 'required|array|min:1',
            'products.*.category_id' => 'required|exists:categories,id',
            'products.*.subcategory_id' => 'required|exists:subcategories,id',
            'products.*.quantity_unit_id' => 'required|exists:quantity_units,id',
            'products.*.brand_id' => 'required|exists:brands,id',
            'products.*.price' => 'required|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'errors' => $validator->errors()], 422);
        }

        $registrationToken = Str::uuid()->toString();
        $data = $request->except(['password']);
        $data['password'] = bcrypt($request->password);

        Cache::put(
            'warehouse_register_' . $registrationToken,
            ['data' => $data],
            now()->addMinutes(15)
        );

        $amount = 100;

        $api = new RazorpayApi(
            config('services.razorpay.key'),
            config('services.razorpay.secret')
        );

        $order = $api->order->create([
            'amount' => $amount * 100,
            'currency' => 'INR',
            'receipt' => 'warehouse_reg_' . Str::random(8),
            'payment_capture' => 1,
        ]);

        Cache::put(
            'payment_order_' . $order['id'],
            $registrationToken,
            now()->addMinutes(15)
        );

        return response()->json([
            'success' => true,
            'gateway' => 'razorpay',
            'order_id' => $order['id'],
            'amount' => $amount,
            'registration_token' => $registrationToken
        ]);
    }

    public function completeRegistration(Request $request)
    {
        $request->validate([
            'order_id'         => 'required|string',
            'payment_status'   => 'required|in:success,failed',
            'payment_response' => 'required|array',
        ]);

        $registrationToken = Cache::get('payment_order_' . $request->order_id);

        if (!$registrationToken) {
            return response()->json([
                'success' => false,
                'message' => 'Payment session expired'
            ], 400);
        }

        if ($request->payment_status !== 'success') {
            return response()->json([
                'success' => false,
                'message' => 'Payment failed'
            ], 400);
        }

        try {
            $api = new Api(
                config('services.razorpay.key'),
                config('services.razorpay.secret')
            );

            $api->utility->verifyPaymentSignature([
                'razorpay_order_id'   => $request->payment_response['razorpay_order_id'],
                'razorpay_payment_id' => $request->payment_response['razorpay_payment_id'],
                'razorpay_signature'  => $request->payment_response['razorpay_signature'],
            ]);
        } catch (SignatureVerificationError $e) {
            return response()->json([
                'success' => false,
                'message' => 'Payment verification failed'
            ], 400);
        }

        $cached = Cache::get('warehouse_register_' . $registrationToken);

        if (!$cached || !isset($cached['data'])) {
            return response()->json([
                'success' => false,
                'message' => 'Registration session expired'
            ], 400);
        }

        $data = $cached['data'];

        $warehouse = Warehouse::create([
            'owner_name'               => $data['owner_name'],
            'email'                    => $data['email'],
            'mobile_number'            => $data['mobile_number'],
            'password'                 => $data['password'],
            'warehouse_name'           => $data['warehouse_name'],
            'door_no'                  => $data['door_no'],
            'street_name'              => $data['street_name'],
            'area'                     => $data['area'],
            'city'                     => $data['city'],
            'state_id'                 => $data['state_id'],
            'district_id'              => $data['district_id'],
            'pincode'                  => $data['pincode'],
            'country'                  => $data['country'],
            'latitude'                 => $data['lat'],
            'longitude'                => $data['lon'],
            'pan_number'               => $data['pan_number'],
            'aadhar_number'            => $data['aadhar_number'],
            'business_type'            => $data['business_type'] ?? null,
            'business_status'          => $data['business_status'],
            'gst_number'               => $data['business_status'] === 'registered'
                                            ? $data['gst_number']
                                            : null,
            'trade_license_number'     => $data['business_status'] === 'registered'
                                            ? $data['trade_license_number']
                                            : null,
            'status'                   => 'pending_admin',
            'registration_paid'        => 1,
            'registration_payment_id' => json_encode($request->payment_response),
        ]);

        foreach ($data['products'] as $p) {
            WarehouseProduct::create([
                'warehouse_id'      => $warehouse->id,
                'category_id'       => $p['category_id'],
                'subcategory_id'    => $p['subcategory_id'] ?? null,
                'quantity_unit_id'  => $p['quantity_unit_id'],
                'brand_id'          => $p['brand_id'] ?? null,
                'price'             => $p['price'],
                'status'            => 'pending',
            ]);
        }

        Address::create([
            'user_type'     => 'warehouse',
            'user_id'       => $warehouse->id,
            'name'          => $data['owner_name'],
            'mobile_number' => $data['mobile_number'],
            'door_no'       => $data['door_no'],
            'street_name'   => $data['street_name'],
            'area'          => $data['area'],
            'city'          => $data['city'],
            'district_id'   => $data['district_id'],
            'state_id'      => $data['state_id'],
            'pincode'       => $data['pincode'],
            'lat'           => $data['lat'],
            'lon'           => $data['lon'],
            'is_default'    => 1
        ]);

        Cache::forget('warehouse_register_' . $registrationToken);
        Cache::forget('payment_order_' . $request->order_id);

        return response()->json([
            'success'      => true,
            'message'      => 'Warehouse registration completed successfully',
            'warehouse_id' => $warehouse->id
        ]);
    }

    // ================= LOGIN WITH JWT =================
    public function login(Request $request)
    {
        $request->validate([
            'login' => 'required',
            'password' => 'required|string|min:4'
        ]);

        $warehouse = Warehouse::where('email', $request->login)
            ->orWhere('mobile_number', $request->login)
            ->first();

        if (!$warehouse) {
            return response()->json([
                'success' => false,
                'message' => 'Account not found'
            ], 404);
        }

        if (!Hash::check($request->password, $warehouse->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid password'
            ], 401);
        }

        if ($warehouse->status !== 'active') {

            $message = match ($warehouse->status) {
                'pending_otp'   => 'OTP verification required',
                'pending_admin' => 'Account pending admin approval',
                'rejected'      => 'Account rejected by admin',
                default         => 'Account inactive',
            };

            return response()->json([
                'success' => false,
                'message' => $message
            ], 403);
        }

        if ($request->filled('fcm_token')) {
            DB::table('fcm_tokens')->updateOrInsert(
                [
                    'user_type' => 'warehouse',
                    'user_id'   => $warehouse->id,
                ],
                [
                    'fcm_token'     => $request->fcm_token,
                    'notify_status' => 'active',
                    'device_type'   => $request->device_type ?? 'android',
                    'updated_at'    => now(),
                    'created_at'    => now()
                ]
            );
        }


        $token = JWTAuth::customClaims([
            'exp' => now()->addDays(30)->timestamp
        ])->fromUser($warehouse);

        return response()->json([
            'success' => true,
            'message' => 'Login successful',
            'token' => $token,
            'token_type' => 'bearer',
            'expires_in' => JWTAuth::factory()->getTTL() * 60,  
            'warehouse' => [
                'id' => $warehouse->id,
                'name' => $warehouse->name,
                'email' => $warehouse->email,
                'mobile_number' => $warehouse->mobile_number,
                'city' => $warehouse->city
            ]
        ]);
    }

    // ================= PROFILE (JWT PROTECTED) =================
    public function profile()
    {
        return response()->json([
            'success' => true,
            'warehouse' => auth('warehouse')->user()
        ]);
    }

    public function changePassword(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        $request->validate([
            'current_password'      => 'required|string',
            'new_password'          => 'required|string|min:8|confirmed',
        ]);

        if (!Hash::check($request->current_password, $warehouse->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect.'
            ], 400);
        }

        $warehouse->password = Hash::make($request->new_password);
        $warehouse->save();

        JWTAuth::invalidate(JWTAuth::getToken());

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully. You have been logged out.'
        ]);
    }

    public function accountDetails()
    {
        $warehouse = auth('warehouse')->user();

        return response()->json([
            'success' => true,
            'data' => [
                'owner_name'     => $warehouse->owner_name,
                'warehouse_name' => $warehouse->warehouse_name,
                'mobile_number'  => $warehouse->mobile_number,

                'logo_url' => $warehouse->logo
                    ? url($warehouse->logo)
                    : null,

                'logo_initial' => strtoupper(
                    substr($warehouse->warehouse_name ?? $warehouse->owner_name, 0, 1)
                ),
            ]
        ]);
    }

        public function updateLogo(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        $request->validate([
            'logo' => 'required|image|mimes:jpg,jpeg,png,webp|max:2048'
        ]);

        if ($warehouse->logo && file_exists(public_path($warehouse->logo))) {
            unlink(public_path($warehouse->logo));
        }

        $file = $request->file('logo');
        $filename = 'warehouse_' . $warehouse->id . '_' . time() . '.' . $file->getClientOriginalExtension();

        $uploadPath = 'uploads/warehouse/logos';
        $file->move(public_path($uploadPath), $filename);

        $warehouse->update([
            'logo' => $uploadPath . '/' . $filename
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Logo updated successfully',
            'logo_url' => url($uploadPath . '/' . $filename)
        ]);
    }


    public function ownerDetails(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        return response()->json([
            'success' => true,
            'data' => [
                'owner_name'    => $warehouse->owner_name,
                'mobile_number' => $warehouse->mobile_number,
                'email'         => $warehouse->email,
            ]
        ]);
    }

    public function businessDetails()
    {
        $warehouse = auth('warehouse')->user()->load(['state', 'district']);

        return response()->json([
            'success' => true,
            'data' => [
                'warehouse_name'  => $warehouse->warehouse_name,
                'door_no'         => $warehouse->door_no,
                'street_name'     => $warehouse->street_name,
                'area'            => $warehouse->area,
                'city'            => $warehouse->city,
                'district_name'   => $warehouse->district->name ?? null,
                'state_name'      => $warehouse->state->name ?? null,
                'pincode'         => $warehouse->pincode,
                'business_type'   => $warehouse->business_type,
                'business_status' => $warehouse->business_status,
            ]
        ]);
    }

    public function legalDocuments()
    {
        $warehouse = auth('warehouse')->user();
        return response()->json([
            'success' => true,
            'data' => [
                'gst_number'           => $warehouse->gst_number,
                'trade_license_number' => $warehouse->trade_license_number,
                'pan_number'           => $warehouse->pan_number,
            ]
        ]);
    }

    public function bankDetails()
    {
        $warehouse = auth('warehouse')->user()->load('bank');

        return response()->json([
            'success' => true,
            'bank_status' => $warehouse->bank_status,
            'has_pending_update' => !empty($store->pending_bank_data),

            'data' => [
                'bank_holder_name'    => $warehouse->bank_holder_name,
                'bank_id'             => $warehouse->bank_id,
                'bank_name'           => $warehouse->bank->name ?? null,
                'bank_account_number' => $warehouse->bank_account_number,
                'ifsc_code'           => $warehouse->ifsc_code,
                'micr_no'             => $warehouse->micr_no,
                'upi_id'              => $warehouse->upi_id,
                'passbook_file'       => $warehouse->passbook_file 
                    ? asset('storage/' . $warehouse->passbook_file) 
                    : null,
            ],

            'pending_data' => $warehouse->pending_bank_data,
            'admin_remark' => $warehouse->admin_remark,
        ]);
    }


    public function saveBankDetails(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        if ($warehouse->bank_id) {
            return response()->json([
                'success' => false,
                'message' => 'Bank details already exist. Please update instead.'
            ], 403);
        }

        $request->validate([
            'bank_id' => 'required|exists:banks,id',
            'bank_holder_name' => 'required|string|max:255',
            'bank_account_number' => 'required|confirmed',
            'bank_account_number_confirmation' => 'required',
            'ifsc_code' => 'required|string|max:255',
            'micr_no' => 'nullable|string|max:50',
            'upi_id' => 'nullable|string|max:100',
            'passbook_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:1024',
        ]);

        $file = $request->file('passbook_file');
        $filename = 'passbook_' . $warehouse->id . '_' . time() . '.' . $file->getClientOriginalExtension();
        $uploadPath = 'warehouse/passbooks';
        $file->move(public_path($uploadPath), $filename);

        $warehouse->update([
            'pending_bank_data' => [
                'bank_id' => $request->bank_id,
                'bank_holder_name' => $request->bank_holder_name,
                'bank_account_number' => $request->bank_account_number,
                'ifsc_code' => $request->ifsc_code,
                'micr_no' => $request->micr_no,
                'upi_id' => $request->upi_id,
                'passbook_file' => $uploadPath . '/' . $filename,
            ],
            'bank_status' => 'pending',
            'admin_remark' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Bank details submitted for admin approval'
        ]);
    }

    public function updateBankDetails(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        if ($warehouse->bank_status === 'pending') {
            return response()->json([
                'success' => false,
                'message' => 'Already waiting for admin approval'
            ], 403);
        }

        $request->validate([
            'bank_id' => 'sometimes|exists:banks,id',
            'bank_holder_name' => 'sometimes|string|max:255',
            'bank_account_number' => 'sometimes|confirmed',
            'bank_account_number_confirmation' => 'required_with:bank_account_number',
            'ifsc_code' => 'sometimes|string|max:255',
            'micr_no' => 'nullable|string|max:50',
            'upi_id' => 'nullable|string|max:100',
            'passbook_file' => 'required|file|mimes:jpg,jpeg,png,pdf|max:1024',
        ]);

        $pending = $store->pending_bank_data ?? [];

        if ($request->hasFile('passbook_file')) {
            if (!empty($pending['passbook_file']) && file_exists(public_path($pending['passbook_file']))) {
                unlink(public_path($pending['passbook_file']));
            }

            $file = $request->file('passbook_file');
            $filename = 'passbook_' . $warehouse->id . '_' . time() . '.' . $file->getClientOriginalExtension();
            $uploadPath = 'warehouse/passbooks';
            $file->move(public_path($uploadPath), $filename);

            $pending['passbook_file'] = $uploadPath . '/' . $filename;
        }

        $warehouse->update([
            'pending_bank_data' => [
                'bank_id' => $request->bank_id ?? $warehouse->bank_id,
                'bank_holder_name' => $request->bank_holder_name ?? $warehouse->bank_holder_name,
                'bank_account_number' => $request->bank_account_number ?? $warehouse->bank_account_number,
                'ifsc_code' => $request->ifsc_code ?? $warehouse->ifsc_code,
                'micr_no' => $request->micr_no ?? $warehouse->micr_no,
                'upi_id' => $request->upi_id ?? $warehouse->upi_id,
                'passbook_file' => $pending['passbook_file'] ?? $warehouse->passbook_file,
            ],
            'bank_status' => 'pending',
            'admin_remark' => null,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Updated bank details sent for admin approval'
        ]);
    }

    public function quotationDetails()
    {
        $warehouse = auth('warehouse')->user();

        $products = WarehouseProduct::with([
                'category:id,name',
                'subcategory:id,name',
                'brand:id,name',
                'quantityUnit:id,name'
            ])
            ->where('warehouse_id', $warehouse->id)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $products->map(function ($item) {
                return [
                    'category_name'   => $item->category->name ?? null,
                    'sub_category'    => $item->subcategory->name ?? null,
                    'brand_name'      => $item->brand->name ?? null,
                    'unit'            => $item->quantityUnit->name ?? null,
                    'price'           => $item->price,
                    'status'          => $item->status
                ];
            })
        ]);
    }

    public function warehouseCategoryPage()
    {
        $warehouse = auth('warehouse')->user();

        $categories = \DB::table('warehouse_products')
            ->join('categories', 'categories.id', '=', 'warehouse_products.category_id')
            ->where('warehouse_products.warehouse_id', $warehouse->id)
            ->groupBy(
                'warehouse_products.category_id',
                'categories.name',
                'categories.image',
                'warehouse_products.status',
                'warehouse_products.category_active'
            )
            ->select(
                'warehouse_products.category_id',
                'categories.name as category_name',
                'categories.image as category_image',
                'warehouse_products.status as product_status',
                'warehouse_products.category_active'
            )
            ->orderBy('categories.name')
            ->get()
            ->map(function ($item) {
                return [
                    'category_id'     => $item->category_id,
                    'category_name'   => $item->category_name,
                    'category_image'  => $item->category_image ? asset($item->category_image) : asset('images/no-image.png'),
                    'status'          => $item->product_status,
                    'category_status' => $item->category_active == 1 ? 'active' : 'inactive',
                ];
            });

        return response()->json([
            'success' => true,
            'data'    => $categories
        ], 200);
    }



    // public function categoryProducts(Request $request)
    // {
    //     $warehouse = auth('warehouse')->user();

    //     $request->validate([
    //         'category' => 'required|string'
    //     ]);

    //     $products = \DB::table('warehouse_products')
    //         ->leftJoin('categories', 'categories.id', '=', 'warehouse_products.category_id')
    //         ->leftJoin('subcategories', 'subcategories.id', '=', 'warehouse_products.subcategory_id')
    //         ->leftJoin('brands', 'brands.id', '=', 'warehouse_products.brand_id')
    //         ->where('warehouse_products.warehouse_id', $warehouse->id)
    //         ->where('categories.name', $request->category)
    //         ->select(
    //             'brands.name as brand_name',
    //             'categories.name as category_name',
    //             'subcategories.name as subcategory_name',
    //             'warehouse_products.quantity_unit',
    //             'warehouse_products.price',
    //             'warehouse_products.status'
    //         )
    //         ->orderBy('brands.name')
    //         ->get();

    //     return response()->json([
    //         'success'  => true,
    //         'category' => $request->category,
    //         'data'     => $products
    //     ]);
    // }

    public function getWarehouseProductForEdit($id)
    {
        $warehouse = auth('warehouse')->user();

        $product = \DB::table('warehouse_products as wp')
            ->leftJoin('categories as c', 'c.id', '=', 'wp.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'wp.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'wp.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'wp.quantity_unit_id')
            ->where('wp.id', $id)
            ->where('wp.warehouse_id', $warehouse->id)
            ->select(
                'wp.id',
                'c.name as category',
                's.name as subcategory',
                'b.name as brand',
                'q.name as quantity_unit',
                'wp.price',
                'wp.status'
            )
            ->first();

        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        $statusMessage = $product->status === 'rejected'
            ? 'Waiting for admin approval'
            : $product->status;

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $product->id,
                'category' => $product->category,
                'subcategory' => $product->subcategory,
                'brand' => $product->brand,
                'quantity_unit' => $product->quantity_unit,
                'price' => $product->price,
                'status' => $statusMessage

            ]
        ]);
    }

    public function categoryProductsById(Request $request, $category_id)
    {
        $warehouse = auth('warehouse')->user();

        $query = \DB::table('warehouse_products as wp')
            ->leftJoin('products as p', function ($join) {
                $join->on('p.category_id', '=', 'wp.category_id')
                    ->on('p.quantity_unit_id', '=', 'wp.quantity_unit_id')
                    ->whereColumn('p.subcategory_id', 'wp.subcategory_id')
                    ->whereColumn('p.brand_id', 'wp.brand_id');
            })
            ->leftJoin('categories as c', 'c.id', '=', 'wp.category_id')
            ->leftJoin('subcategories as s', 's.id', '=', 'wp.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'wp.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'wp.quantity_unit_id')
            ->where('wp.warehouse_id', $warehouse->id)
            ->where('wp.category_id', $category_id);

        if ($request->filled('subcategory_id')) {
            $query->where('wp.subcategory_id', $request->subcategory_id);
        }

        if ($request->filled('brand_id')) {
            $query->where('wp.brand_id', $request->brand_id);
        }

        if ($request->filled('quantity_unit_id')) {
            $query->where('wp.quantity_unit_id', $request->quantity_unit_id);
        }

        if ($request->filled('active')) {
            $query->where('wp.product_active', $request->active);
        }

        $products = $query
            ->select(
                'wp.id as warehouse_product_id',
                'c.id as category_id',
                'c.name as category_name',
                's.name as subcategory_name',
                'b.name as brand_name',
                'q.name as quantity_unit',
                'wp.price',
                'wp.status',
                'wp.product_active',
                'wp.admin_remark',
                'p.image as product_image'
            )
            ->orderBy('wp.created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $products->map(function ($item) {
                return [
                    'id' => $item->warehouse_product_id,
                    'category_id' => $item->category_id,
                    'category' => $item->category_name,
                    'subcategory' => $item->subcategory_name,
                    'brand' => $item->brand_name,
                    'unit' => $item->quantity_unit,
                    'price' => $item->price,
                    'status' => $item->status,
                    'admin_remark' => $item->admin_remark,
                    'active' => (bool) $item->product_active,
                    'image' => $item->product_image
                        ? asset($item->product_image)
                        : null
                ];
            })
        ]);
    }



    public function updateWarehouseProduct(Request $request, $id)
    {
        $warehouse = auth('warehouse')->user();

        $request->validate([
            'price' => 'required|numeric|min:0',
        ]);

        $warehouseProduct = \DB::table('warehouse_products')
            ->where('id', $id)
            ->where('warehouse_id', $warehouse->id)
            ->first();

        if (!$warehouseProduct) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        \DB::table('warehouse_products')
            ->where('id', $id)
            ->update([
                'price' => $request->price,
                'status' => 'pending',
                'admin_remark' => null,
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Price updated and sent for admin approval'
        ]);
    }


    public function addProducts(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        $request->validate([
            'products' => 'required|array|min:1',
            'products.*.category_id' => 'required|exists:categories,id',
            'products.*.subcategory_id' => 'nullable|exists:subcategories,id',
            'products.*.quantity_unit_id' => 'required|exists:quantity_units,id',
            'products.*.brand_id' => 'nullable|exists:brands,id',
            'products.*.price' => 'required|numeric|min:0',
        ]);

        foreach ($request->products as $p) {

            $validUnit = QuantityUnit::where('id', $p['quantity_unit_id'])
                ->whereJsonContains('category_ids', (string) $p['category_id'])
                ->exists();

            if (!$validUnit) {
                continue;
            }
            $productQuery = Product::where('status', 'active')
                ->where('category_id', $p['category_id'])
                ->where('quantity_unit_id', $p['quantity_unit_id']);

            if (!empty($p['subcategory_id'])) {
                $productQuery->where('subcategory_id', $p['subcategory_id']);
            } else {
                $productQuery->whereNull('subcategory_id');
            }

            if (!empty($p['brand_id'])) {
                $productQuery->where('brand_id', $p['brand_id']);
            } else {
                $productQuery->whereNull('brand_id');
            }

            $product = $productQuery->first();

            WarehouseProduct::create([
                'warehouse_id'     => $warehouse->id,
                'product_id'       => $product?->id,
                'category_id'      => $p['category_id'],
                'subcategory_id'   => $p['subcategory_id'] ?? null,
                'quantity_unit_id' => $p['quantity_unit_id'],
                'brand_id'         => $p['brand_id'] ?? null,
                'price'            => $p['price'],
                'status'           => 'pending',
            ]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Product(s) added and sent for admin approval'
        ]);
    }

    public function toggleWarehouseCategoryActive($category_id)
    {
        $warehouse = auth('warehouse')->user();

        // Get current state
        $current = \DB::table('warehouse_products')
            ->where('warehouse_id', $warehouse->id)
            ->where('category_id', $category_id)
            ->value('category_active');

        if ($current === null) {
            return response()->json([
                'success' => false,
                'message' => 'Category not found'
            ], 404);
        }

        $newStatus = !$current;

        \DB::table('warehouse_products')
            ->where('warehouse_id', $warehouse->id)
            ->where('category_id', $category_id)
            ->update([
                'category_active' => $newStatus
            ]);

        return response()->json([
            'success' => true,
            'active' => $newStatus
        ]);
    }

    public function deleteWarehouseCategory($category_id)
    {
        $warehouse = auth('warehouse')->user();

        \DB::table('warehouse_products')
            ->where('warehouse_id', $warehouse->id)
            ->where('category_id', $category_id)
            ->delete();

        return response()->json([
            'success' => true,
            'message' => 'Category removed'
        ]);
    }

    public function toggleWarehouseProductActive($warehouse_product_id)
    {
        $warehouse = auth('warehouse')->user();

        // Get current state
        $current = \DB::table('warehouse_products')
            ->where('warehouse_id', $warehouse->id)
            ->where('id', $warehouse_product_id)
            ->value('product_active');

        if ($current === null) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        $newStatus = !$current;

        \DB::table('warehouse_products')
            ->where('warehouse_id', $warehouse->id)
            ->where('id', $warehouse_product_id)
            ->update([
                'product_active' => $newStatus
            ]);

        return response()->json([
            'success' => true,
            'active'  => $newStatus
        ]);
    }

    public function deleteWarehouseProduct($warehouse_product_id)
    {
        $warehouse = auth('warehouse')->user();

        $deleted = \DB::table('warehouse_products')
            ->where('warehouse_id', $warehouse->id)
            ->where('id', $warehouse_product_id)
            ->delete();

        if (!$deleted) {
            return response()->json([
                'success' => false,
                'message' => 'Product not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Product removed'
        ]);
    }

    public function categoryProductFilters($category_id)
    {
        $warehouse = auth('warehouse')->user();

        $base = \DB::table('warehouse_products')
            ->where('warehouse_products.warehouse_id', $warehouse->id)
            ->where('warehouse_products.category_id', $category_id);

        return response()->json([
            'success' => true,
            'filters' => [

                'subcategories' => (clone $base)
                    ->join('subcategories', 'subcategories.id', '=', 'warehouse_products.subcategory_id')
                    ->select('subcategories.id', 'subcategories.name')
                    ->distinct()
                    ->orderBy('subcategories.name')
                    ->get(),

                'brands' => (clone $base)
                    ->join('brands', 'brands.id', '=', 'warehouse_products.brand_id')
                    ->select('brands.id', 'brands.name')
                    ->distinct()
                    ->orderBy('brands.name')
                    ->get(),

                'quantity_units' => (clone $base)
                    ->join('quantity_units', 'quantity_units.id', '=', 'warehouse_products.quantity_unit_id')
                    ->select('quantity_units.id', 'quantity_units.name')
                    ->distinct()
                    ->orderBy('quantity_units.name')
                    ->get(),

                'statuses' => [
                    ['id' => 1, 'name' => 'Active'],
                    ['id' => 0, 'name' => 'Inactive'],
                ],
            ]
        ]);
    }


    public function logout(Request $request)
    {
        try {
            $user = auth('warehouse')->user();

            if ($user) {
                DB::table('fcm_tokens')
                    ->where('user_type', 'warehouse')
                    ->where('user_id', $user->id)
                    ->update([
                        'notify_status' => 'inactive',
                        'updated_at'    => now()
                    ]);
            }

            JWTAuth::invalidate(JWTAuth::getToken());

            return response()->json([
                'success' => true,
                'message' => 'Logged out successfully'
            ]);

        } catch (JWTException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to logout',
                'error'   => $e->getMessage()
            ], 500);
        }
    }



    public function refresh()
    {
        try {
            $newToken = JWTAuth::refresh(JWTAuth::getToken());

            return response()->json([
                'success' => true,
                'token' => $newToken
            ]);
        } catch (JWTException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Token refresh failed'
            ], 401);
        }
    }


    // ================= FORGOT PASSWORD - SEND OTP =================
    public function forgotPassword(Request $request)
    {
        $request->validate([
            'mobile_number' => 'required|digits:10'
        ]);

        $warehouse = Warehouse::where('mobile_number', $request->mobile_number)->first();

        if (!$warehouse) {
            return response()->json([
                'success' => false,
                'message' => 'Mobile number not registered'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'message' => 'Mobile number verified. Proceed with OTP.'
        ]);
    }
    // ================= FORGOT PASSWORD - VERIFY OTP =================

    public function verifyForgotOtp(Request $request)
    {
        $request->validate([
            'mobile_number'  => 'required|digits:10',
            'firebase_token' => 'required'
        ]);

        try {
            $factory = (new Factory)->withServiceAccount(
                base_path(env('FIREBASE_CREDENTIALS'))
            );

            $auth = $factory->createAuth();

            $verifiedToken = $auth->verifyIdToken($request->firebase_token);
            $firebasePhone = $verifiedToken->claims()->get('phone_number');
            $requestPhone  = '+91' . $request->mobile_number;

            if ($firebasePhone !== $requestPhone) {
                return response()->json([
                    'success' => false,
                    'message' => 'Mobile number mismatch'
                ], 401);
            }

            $warehouse = Warehouse::where('mobile_number', $request->mobile_number)->first();

            if (!$warehouse) {
                return response()->json([
                    'success' => false,
                    'message' => 'User not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'message' => 'OTP verified successfully',
                'warehouse_id' => $warehouse->id
            ]);

        } catch (\Throwable $e) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid or expired OTP'
            ], 401);
        }
    }

    // ================= RESET PASSWORD =================
    public function resetPassword(Request $request)
    {
        $request->validate([
            'mobile_number' => 'required|digits:10|exists:warehouses,mobile_number',
            'password'      => 'required|min:8|confirmed',
        ]);

        Warehouse::where('mobile_number', $request->mobile_number)->update([
            'password' => Hash::make($request->password),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password reset successfully'
        ]);
    }


    public function notifications(Request $request)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $notifications = DB::table('notifications')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('created_at', '>=', Carbon::now()->subDays(15))
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function($n){
                $link = $n->link ? json_decode($n->link, true) : null;

                return [
                    'id'         => $n->id,
                    'title'      => $n->title,
                    'content'    => $n->content,
                    'route'      => $link['route'] ?? null,
                    'route_id'   => $link['route_id'] ?? null,
                    'status'     => $n->status,
                    'created_date'=> Carbon::parse($n->created_at)->format('Y-m-d'),
                    'time'       => Carbon::parse($n->created_at)->format('H:i:s'),
                    'time_ago'   => Carbon::parse($n->created_at)->diffForHumans(),
                ];
            });

        return response()->json([
            'success' => true,
            'count'   => $notifications->count(),
            'data'    => $notifications
        ]);
    }

    public function activeNotificationsCount(Request $request)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $activeCount = DB::table('notifications')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('status', 'active')
            ->count();

        return response()->json([
            'success' => true,
            'active_count' => $activeCount
        ]);
    }

    public function toggleNotificationStatus(Request $request, $notificationId)
    {
        if (auth('warehouse')->check()) {
            $userType = 'warehouse';
            $userId   = auth('warehouse')->id();
        } elseif (auth('store')->check()) {
            $userType = 'store';
            $userId   = auth('store')->id();
        } elseif (auth('rider')->check()) {
            $userType = 'rider';
            $userId   = auth('rider')->id();
        } elseif (auth('customer')->check()) {
            $userType = 'customer';
            $userId   = auth('customer')->id();
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $notification = DB::table('notifications')
            ->where('id', $notificationId)
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->first();

        if ($notification->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Notification is already inactive'
            ], 400);
        }

        DB::table('notifications')
            ->where('id', $notificationId)
            ->update([
                'status'     => 'inactive',
                'updated_at' => now(),
            ]);

        return response()->json([
            'success' => true,
            'message' => 'Notification deactivated successfully',
            'data' => [
                'notification_id' => $notificationId,
                'status' => 'inactive',
            ]
        ], 200);

    }
}


