<?php

namespace App\Http\Controllers;
use App\Models\Store;
use App\Models\Warehouse;
use App\Mail\OrderTakenMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Carbon\Carbon;


class WarehouseOrderController extends Controller
{
    private function sendNotification($title, $content, $userType, $userId, $route = null, $routeId = null)
    {
        $linkData = null;

        if ($route) {
            $linkData = $routeId !== null
                ? json_encode(['route' => $route, 'route_id' => $routeId], JSON_UNESCAPED_SLASHES)
                : json_encode(['route' => $route], JSON_UNESCAPED_SLASHES);
        }


        DB::table('notifications')->insert([
            'title'      => $title,
            'content'    => $content,
            'user_type'  => $userType,
            'user_id'    => $userId,
            'status'     => 'active',
            'link'       => $linkData,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        /* ========== 2. Get active FCM tokens ========== */
        $tokens = DB::table('fcm_tokens')
            ->where('user_type', $userType)
            ->where('user_id', $userId)
            ->where('notify_status', 'active')
            ->pluck('fcm_token')
            ->toArray();

        if (empty($tokens)) {
            return; 
        }

        /* ========== 3. Firebase HTTP v1 Auth ========== */
        $credentialsPath = base_path(env('FIREBASE_CREDENTIALS'));
        $projectId       = env('FIREBASE_PROJECT_ID');

        $client = new \Google_Client();
        $client->setAuthConfig($credentialsPath);
        $client->addScope('https://www.googleapis.com/auth/firebase.messaging');

        $accessToken = $client->fetchAccessTokenWithAssertion();

        if (!isset($accessToken['access_token'])) {
            \Log::error('Firebase token error', $accessToken);
            return;
        }

        $bearerToken = $accessToken['access_token'];

        $url = "https://fcm.googleapis.com/v1/projects/{$projectId}/messages:send";

        foreach ($tokens as $fcmToken) {

            $dataPayload = [
                "route"     => (string) $route,
                "type"      => "notification",
                "user_type" => (string) $userType,
                "status"    => "active",
            ];

            if ($routeId !== null) {
                $dataPayload["route_id"] = (string) $routeId;
            }


            $payload = [
                "message" => [
                    "token" => $fcmToken,
                    "notification" => [
                        "title" => $title,
                        "body"  => $content,
                    ],
                    "data" => $dataPayload,
                    "android" => [
                        "priority" => "high",
                        "notification" => [
                            "sound" => "default",
                            "click_action" => "FLUTTER_NOTIFICATION_CLICK"
                        ]
                    ]
                ]
            ];

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                "Authorization: Bearer {$bearerToken}",
                "Content-Type: application/json"
            ]);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));

            $response = curl_exec($ch);

            if ($response === false) {
                \Log::error('Firebase Push Error: ' . curl_error($ch));
            }

            curl_close($ch);
        }
    }

    public function warehouseDashboard(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        if (!$warehouse) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        $today = Carbon::today();

        $lastWeekStart = Carbon::now()->subWeek()->startOfWeek(Carbon::MONDAY);
        $lastWeekEnd   = Carbon::now()->subWeek()->endOfWeek(Carbon::SUNDAY);

        $todayOrderTakenCount = DB::table('orders')
            ->where('approved_by', $warehouse->id)
            ->where('approved_by_user_type', 'warehouse')
            ->whereDate('approved_at', $today)
            ->count();

        $todayLoadedOrderCount = DB::table('orders')
            ->where('approved_by', $warehouse->id)
            ->where('approved_by_user_type', 'warehouse')
            ->whereDate('loaded_at', $today)
            ->count();

        $lastWeekRevenue = DB::table('orders as o')
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->where('o.status', 'completed')
            ->whereBetween('o.approved_at', [$lastWeekStart, $lastWeekEnd])
            ->select(DB::raw('SUM(oi.approved_unit_price * oi.quantity) as revenue'))
            ->value('revenue');

        $highestCategory = DB::table('orders as o')
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->join('warehouse_products as wp', function ($join) use ($warehouse) {
                $join->on('wp.product_id', '=', 'oi.product_id')
                    ->where('wp.warehouse_id', '=', $warehouse->id);
            })
            ->select('wp.category_id', DB::raw('SUM(oi.total_price) as total_amount'))
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->where('o.status', 'completed')
            ->whereBetween('o.approved_at', [$lastWeekStart, $lastWeekEnd])
            ->groupBy('wp.category_id')
            ->orderByDesc('total_amount')
            ->first();

        $highestCategoryData = null;

        if ($highestCategory && $highestCategory->category_id) {
            $category = DB::table('categories')
                ->where('id', $highestCategory->category_id)
                ->first();

            $highestCategoryData = [
                'category_id'   => $highestCategory->category_id,
                'category_name' => $category->name ?? null,
                'amount'        => round($highestCategory->total_amount, 2)
            ];
        }

        $pendingOrdersCount = DB::table('orders')
            ->where('approved_by', $warehouse->id)
            ->where('approved_by_user_type', 'warehouse')
            ->where('status', 'order_taken')
            ->count();

        return response()->json([
            'success' => true,
            'warehouse_id' => $warehouse->id,
            'data' => [
                'today' => [
                    'order_taken_count'  => $todayOrderTakenCount,
                    'loaded_order_count' => $todayLoadedOrderCount,
                ],

                'last_week' => [
                    'from' => $lastWeekStart->format('Y-m-d'),
                    'to'   => $lastWeekEnd->format('Y-m-d'),
                    'total_revenue' => round($lastWeekRevenue, 2),
                    'highest_category' => $highestCategoryData
                ],

                'pending_orders_count' => $pendingOrdersCount
            ]
        ]);
    }

    public function getNearbyStoreOrders(Request $request)
    {
        $warehouse = auth('warehouse')->user();
        $radiusKm = $request->input('radius', 10);

        if (!$warehouse->latitude || !$warehouse->longitude) {
            return response()->json([
                'success' => false,
                'message' => 'Warehouse location not available'
            ], 400);
        }

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.rider_id',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name',
                'a.latitude',
                'a.longitude',
                'o.rejected_by',

                DB::raw("
                    IF(
                        o.with_shipping = 1,
                        (
                            6371 * acos(
                                cos(radians(?)) *
                                cos(radians(a.latitude)) *
                                cos(radians(a.longitude) - radians(?)) +
                                sin(radians(?)) *
                                sin(radians(a.latitude))
                            )
                        ),
                        NULL
                    ) AS distance
                ")
            )
            ->addBinding($warehouse->latitude)
            ->addBinding($warehouse->longitude)
            ->addBinding($warehouse->latitude)
            ->where('o.user_type', 'store')
            ->where('o.status', 'pending_assignment')

            ->whereExists(function ($q) use ($warehouse) {
                $q->select(DB::raw(1))
                    ->from('order_items as oi')
                    ->join('warehouse_products as wp', 'wp.product_id', '=', 'oi.product_id')
                    ->whereRaw('oi.order_id = o.id')
                    ->where('wp.warehouse_id', $warehouse->id)
                    ->where('wp.product_active', 1)
                    ->where('wp.status', 'approved');
            })

            ->where(function ($query) use ($warehouse) {
                $currentUserJson = json_encode([
                    'user_id' => $warehouse->id,
                    'user_type' => 'warehouse'
                ]);

                $query->whereNull('o.rejected_by')
                    ->orWhereRaw("JSON_CONTAINS(o.rejected_by, ?, '$') = 0", [$currentUserJson]);
            })


            ->havingRaw("
                (o.with_shipping = 1 AND distance <= ?)
                OR
                (o.with_shipping = 0)
            ", [$radiusKm])
            ->orderByRaw("IF(o.payment_method = 'COD', o.created_at, o.paid_at) DESC")

            ->get();

        $orders = $orders->map(function ($order) {

            $dateTime = $order->payment_method === 'COD'
                ? $order->created_at
                : $order->paid_at;

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'oi.id',
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category' => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand' => $item->brand,
                        'quantity' => $item->quantity,
                        'unit' => $item->quantity_unit,
                        'image' => $item->image
                            ? asset($item->image)
                            : null
                    ];
                });


            return [
                'order_id'   => $order->id,
                'order_no'   => $order->order_no,
                'status'     => $order->status,
                'shipping_type'  => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'rider_id'   => $order->rider_id,
                'rider_code' => $order->rider_code,

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'door_no'      => $order->with_shipping ? $order->door_no : null,
                'street_name'  => $order->with_shipping ? $order->street_name : null,
                'area'         => $order->with_shipping ? $order->area : null,
                'city'         => $order->with_shipping ? $order->city : null,
                'district'     => $order->with_shipping ? $order->district_name : null,
                'state'        => $order->with_shipping ? $order->state_name : null,
                'pincode'      => $order->with_shipping ? $order->pincode : null,

                'distance_km' => $order->distance,

                'order_items' => $items
            ];
        });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    // public function approveOrder($order_id)
    // {
    //     $warehouse = auth('warehouse')->user();

    //     $order = DB::table('orders')->where('id', $order_id)->first();

    //     if (!$order) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Order not found'
    //         ], 404);
    //     }

    //     if ($order->with_shipping == 0) {

    //         if (!$order->rider_id) {
    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Rider not assigned for this order'
    //             ], 400);
    //         }

    //         DB::beginTransaction();

    //         try {
    //             DB::table('orders')->where('id', $order_id)->update([
    //                 'status' => 'rider_accepted',
    //                 'approved_by' => $warehouse->id,
    //                 'approved_by_user_type' => 'warehouse',
    //                 'approved_at' => now(),
    //                 'updated_at' => now(),
    //             ]);

    //             $riderRequest = DB::table('order_rider_requests')
    //                 ->where('order_id', $order_id)
    //                 ->where('rider_id', $order->rider_id)
    //                 ->first();

    //             if ($riderRequest) {
    //                 DB::table('order_rider_requests')
    //                     ->where('id', $riderRequest->id)
    //                     ->update([
    //                         'status' => 'accepted',
    //                         'responded_at' => now(),
    //                         'updated_at' => now(),
    //                     ]);
    //             } else {
    //                 DB::table('order_rider_requests')->insert([
    //                     'order_id'     => $order_id,
    //                     'rider_id'     => $order->rider_id,
    //                     'warehouse_id' => $warehouse->id,
    //                     'status'       => 'accepted',
    //                     'requested_at' => now(),
    //                     'responded_at' => now(),
    //                     'created_at'   => now(),
    //                     'updated_at'   => now(),
    //                 ]);
    //             }

    //             DB::commit();

    //             return response()->json([
    //                 'success' => true,
    //                 'message' => 'Order approved and rider accepted successfully'
    //             ]);

    //         } catch (\Exception $e) {
    //             DB::rollBack();

    //             return response()->json([
    //                 'success' => false,
    //                 'message' => 'Something went wrong',
    //                 'error' => $e->getMessage()
    //             ], 500);
    //         }
    //     }

    //     DB::table('orders')->where('id', $order_id)->update([
    //         'status' => 'order_taken',
    //         'approved_by' => $warehouse->id,
    //         'approved_by_user_type' => 'warehouse',
    //         'approved_at' => now(),
    //         'updated_at' => now(),
    //     ]);

    //     return response()->json([
    //         'success' => true,
    //         'message' => 'Order approved successfully'
    //     ]);
    // }

    private function calculateDistanceKm($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371;

        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);

        $a = sin($dLat / 2) * sin($dLat / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($dLon / 2) * sin($dLon / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));

        return round($earthRadius * $c, 2);
    }
    
    private function sendOrderTakenMail($order)
    {
        if (!$order || !$order->approved_by || !$order->approved_by_user_type) {
            return;
        }

        $receiver = null;

        if ($order->approved_by_user_type === 'warehouse') {
            $receiver = Warehouse::find($order->approved_by);
        } elseif ($order->approved_by_user_type === 'store') {
            $receiver = Store::find($order->approved_by);
        }

        if (!$receiver || empty($receiver->email)) {
            return;
        }

        Mail::to($receiver->email)->send(
            new OrderTakenMail($order, $receiver, $order->approved_by_user_type)
        );
    }
    public function approveOrder($order_id)
    {
        $warehouse = auth('warehouse')->user();

        $order = DB::table('orders')->where('id', $order_id)->first();
        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        DB::beginTransaction();

        try {
            if ($order->with_shipping == 0) {

                if (!$order->rider_id) {
                    DB::rollBack();
                    return response()->json([
                        'success' => false,
                        'message' => 'Rider not assigned'
                    ], 400);
                }

                $rider = DB::table('riders')->where('id', $order->rider_id)->first();
                if (!$rider || !$rider->latitude || !$rider->longitude) {
                    throw new \Exception('Rider location not available');
                }

                if (!$warehouse->latitude || !$warehouse->longitude) {
                    throw new \Exception('Warehouse location not available');
                }

                $distanceKm = $this->calculateDistanceKm(
                    $rider->latitude,
                    $rider->longitude,
                    $warehouse->latitude,
                    $warehouse->longitude
                );

                $otp = rand(1000, 9999);

                DB::table('orders')->where('id', $order_id)->update([
                    'status'                => 'rider_accepted',
                    'approved_by'           => $warehouse->id,
                    'approved_by_user_type' => 'warehouse',
                    'approved_at'           => now(),
                    'updated_at'            => now(),
                ]);

                
                DB::table('order_rider_requests')->updateOrInsert(
                    [
                        'order_id' => $order_id,
                        'rider_id' => $order->rider_id
                    ],
                    [
                        'warehouse_id' => $warehouse->id,
                        'latitude'     => $rider->latitude,
                        'longitude'    => $rider->longitude,
                        'distance_km'  => $distanceKm,
                        'status'       => 'active',
                        'otp'          => $otp,
                        'accepted_at'  => now(),
                        'requested_at' => now(),
                        'responded_at' => now(),
                        'updated_at'   => now(),
                    ]
                );

                DB::table('riders')
                    ->where('id', $order->rider_id)
                    ->update([
                        'work_status' => 1,
                        'updated_at'  => now(),
                    ]);
                

                DB::commit();

                $orderFresh = DB::table('orders')->where('id', $order_id)->first();
                $this->sendOrderTakenMail($orderFresh);

                return response()->json([
                    'success'     => true,
                    'message'     => 'Order approved (Without Shipping)',
                    'order_id'    => $order_id,
                    'otp'         => $otp,
                    'distance_km' => $distanceKm
                ]);
            }

            DB::table('orders')->where('id', $order_id)->update([
                'status'                => 'order_taken',
                'approved_by'           => $warehouse->id,
                'approved_by_user_type' => 'warehouse',
                'approved_at'           => now(),
                'updated_at'            => now(),
            ]);

            $orderItem = DB::table('order_items')
                ->where('order_id', $order_id)
                ->first();

            if (!$orderItem) {
                throw new \Exception('Order item not found');
            }

            $warehouseProduct = DB::table('warehouse_products')
                ->where('product_id', $orderItem->product_id)
                ->where('warehouse_id',  $warehouse->id)
                ->first();

            if (!$warehouseProduct) {
                throw new \Exception(
                    "Warehouse product not found for product_id={$orderItem->product_id}, warehouse_id={ $warehouse->id}"
                );
            }

            DB::table('order_items')
                ->where('id', $orderItem->id)
                ->update([
                    'approved_unit_price' => $warehouseProduct->price,
                    'total_price'         => $warehouseProduct->price * $orderItem->quantity,
                    'updated_at'          => now()
                ]);

            DB::commit();

            $orderFresh = DB::table('orders')->where('id', $order_id)->first();
            $this->sendOrderTakenMail($orderFresh);


            return response()->json([
                'success'  => true,
                'message'  => 'Order approved (With Shipping)',
                'order_id' => $order_id
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function rejectOrder(Request $request, $order_id)
    {
        $warehouse = auth('warehouse')->user();

        $order = DB::table('orders')->where('id', $order_id)->first();
        if (!$order) {
            return response()->json(['success' => false, 'message' => 'Order not found'], 404);
        }

        $newEntry = [
            'user_id' => $warehouse->id,
            'user_type' => 'warehouse',
            'rejected_at' => Carbon::now()->toDateTimeString()
        ];

        $existing = $order->rejected_by ? json_decode($order->rejected_by, true) : [];

        $existing[] = $newEntry;

        DB::table('orders')->where('id', $order_id)->update([
            'rejected_by' => json_encode($existing),
            'updated_at' => Carbon::now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Order rejected successfully'
        ]);
    }

    public function getPendingOrders(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.rider_id',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name',
                'a.latitude',
                'a.longitude'
            )
            ->where('o.user_type', 'store')
            ->where('o.status', 'order_taken')
            ->where('o.with_shipping', 1)
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->orderByRaw("IF(o.payment_method = 'COD', o.created_at, o.paid_at) DESC")
            ->get();

            $orders = $orders->map(function ($order) {
            $dateTime = $order->payment_method === 'COD' ? $order->created_at : $order->paid_at;

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'oi.id',
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category' => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand' => $item->brand,
                        'quantity' => $item->quantity,
                        'unit' => $item->quantity_unit,
                        'image' => $item->image ? asset($item->image) : null
                    ];
                });

            return [
                'order_id'   => $order->id,
                'order_no'   => $order->order_no,
                'status'     => $order->status,
                'shipping_type'  => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'rider_id'   => $order->rider_id,
                'rider_code' => $order->rider_code,

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'door_no'     => $order->door_no,
                'street_name' => $order->street_name,
                'area'        => $order->area,
                'city'        => $order->city,
                'district'    => $order->district_name,
                'state'       => $order->state_name,
                'pincode'     => $order->pincode,

                'order_items' => $items
            ];
        });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getAvailableRiders($order_id)
    {
        $warehouse = auth('warehouse')->user();
        $radiusKm = 10;

        if (!$warehouse->latitude || !$warehouse->longitude) {
            return response()->json([
                'success' => false,
                'message' => 'Warehouse location not available'
            ], 400);
        }

        $order = DB::table('orders')->where('id', $order_id)->first();
        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $lastRequestAt = DB::table('order_rider_requests')
            ->where('order_id', $order_id)
            ->max('requested_at');

        $enableSendNearby = true;
        $remainingMinutes = 0;

        if ($lastRequestAt) {
            $nextAllowedTime = Carbon::parse($lastRequestAt)->addHours(2);
            $now = now();

            if ($now->lessThan($nextAllowedTime)) {
                $enableSendNearby = false;
                $remainingMinutes = $now->diffInMinutes($nextAllowedTime);
            }
        }

        $riders = collect();

        if ($enableSendNearby) {
            $riders = DB::table('riders as r')
                ->join('rider_locations as rl', 'rl.rider_id', '=', 'r.id')
                ->select(
                    'r.id',
                    'r.rider_code',
                    'r.name',
                    'rl.latitude',
                    'rl.longitude',
                    DB::raw("(
                        6371 * acos(
                            cos(radians(?)) *
                            cos(radians(rl.latitude)) *
                            cos(radians(rl.longitude) - radians(?)) +
                            sin(radians(?)) *
                            sin(radians(rl.latitude))
                        )
                    ) AS distance")
                )
                ->addBinding($warehouse->latitude)
                ->addBinding($warehouse->longitude)
                ->addBinding($warehouse->latitude)
                ->where('r.status', 'active')
                ->where('r.is_available', 1)
                ->whereNotNull('rl.latitude')
                ->whereNotNull('rl.longitude')
                ->having('distance', '<=', $radiusKm)
                ->orderBy('distance', 'asc')
                ->get();
        }

        return response()->json([
            'success' => true,
            'enable_send_nearby' => $enableSendNearby,
            'remaining_minutes' => $remainingMinutes,
            'count' => $riders->count(),
            'data'  => $riders
        ]);
    }

    public function sendRiderRequests(Request $request, $order_id)
    {
        $warehouse = auth('warehouse')->user();
        $radiusKm = 10;

        $request->validate([
            'rider_ids'   => 'required|array|min:1',
            'rider_ids.*' => 'exists:riders,id'
        ]);

        $order = DB::table('orders as o')
            ->leftJoin('addresses as a', 'a.id', '=', 'o.shipping_address_id')
            ->leftJoin('warehouses as w', function($join) {
                $join->on('w.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'warehouse');
            })
            ->leftJoin('stores as s', function($join) {
                $join->on('s.id', '=', 'o.approved_by')
                    ->where('o.approved_by_user_type', 'store');
            })
            ->select(
                'o.*',
                'a.latitude as shipping_lat',
                'a.longitude as shipping_lon',
                'w.latitude as warehouse_lat',
                'w.longitude as warehouse_lon',
                's.latitude as store_lat',
                's.longitude as store_lon'
            )
            ->where('o.id', $order_id)
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $haversine = function($lat1, $lon1, $lat2, $lon2) {
            $earthRadius = 6371; // km

            $latFrom = deg2rad($lat1);
            $lonFrom = deg2rad($lon1);
            $latTo   = deg2rad($lat2);
            $lonTo   = deg2rad($lon2);

            $latDelta = $latTo - $latFrom;
            $lonDelta = $lonTo - $lonFrom;

            $angle = 2 * asin(sqrt(
                pow(sin($latDelta / 2), 2) +
                cos($latFrom) * cos($latTo) *
                pow(sin($lonDelta / 2), 2)
            ));

            return $angle * $earthRadius;
        };

        DB::beginTransaction();
        try {
            foreach ($request->rider_ids as $riderId) {

                $rider = DB::table('riders as r')
                    ->join('rider_locations as rl', 'rl.rider_id', '=', 'r.id')
                    ->select(
                        'r.id',
                        'r.is_available',
                        'r.ratePerKm',
                        'rl.latitude',
                        'rl.longitude'
                    )
                    ->where('r.id', $riderId)
                    ->where('r.status', 'active')
                    ->where('r.is_available', 1)
                    ->whereNotNull('rl.latitude')
                    ->whereNotNull('rl.longitude')
                    ->first();

                if (!$rider) continue;

                $approvedLat = $order->approved_by_user_type === 'warehouse'
                    ? $order->warehouse_lat
                    : $order->store_lat;

                $approvedLon = $order->approved_by_user_type === 'warehouse'
                    ? $order->warehouse_lon
                    : $order->store_lon;

                if (!$approvedLat || !$approvedLon || !$order->shipping_lat || !$order->shipping_lon) {
                    continue; // skip if any coords missing
                }

                $distanceKm = round($haversine($order->shipping_lat, $order->shipping_lon, $approvedLat, $approvedLon), 2);
                $tripAmount = round($distanceKm * $rider->ratePerKm, 2);

                $existingRequest = DB::table('order_rider_requests')
                    ->where('order_id', $order_id)
                    ->where('rider_id', $riderId)
                    ->first();

                if ($existingRequest) {
                    $nextAllowedTime = Carbon::parse($existingRequest->requested_at)->addHours(2);
                    if (now()->lessThan($nextAllowedTime)) continue;

                    DB::table('order_rider_requests')
                        ->where('id', $existingRequest->id)
                        ->update([
                            'status'       => 'requested',
                            'requested_at' => now(),
                            'responded_at' => null,
                            'accepted_at'  => null,
                            'otp'          => null,
                            'distance_km'  => $distanceKm,
                            'trip_amount'  => $tripAmount,
                            'updated_at'   => now(),
                        ]);
                } else {
                    DB::table('order_rider_requests')->insert([
                        'order_id'     => $order_id,
                        'rider_id'     => $riderId,
                        'warehouse_id' => $order->approved_by_user_type === 'warehouse' ? $order->approved_by : null,
                        'store_id'     => $order->approved_by_user_type === 'store' ? $order->approved_by : null,
                        'latitude'     => $order->shipping_lat,
                        'longitude'    => $order->shipping_lon,
                        'distance_km'  => $distanceKm,
                        'trip_amount'  => $tripAmount,
                        'status'       => 'requested',
                        'requested_at' => now(),
                        'created_at'   => now(),
                        'updated_at'   => now(),
                    ]);
                }

                $this->sendNotification(
                    'New Pickup Request',
                    'A nearby delivery request is available. Tap to view and respond.',
                    'rider',
                    $rider->id,
                    'rider/orders/requests',
                    null
                );
            }

            DB::table('orders')
                ->where('id', $order_id)
                ->update([
                    'status'     => 'waiting_for_rider_approval',
                    'updated_at' => now()
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Pickup request sent successfully'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }


    public function getAssignedOrders(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name'
            )
            ->where('o.user_type', 'store')
            ->whereIn('o.status', [
                'waiting_for_rider_approval',
                'rider_accepted'
            ])
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->orderByRaw("IF(o.payment_method = 'COD', o.created_at, o.paid_at) DESC")
            ->get();

            $orders = $orders->map(function ($order) {
                $dateTime = $order->payment_method === 'COD'
                    ? $order->created_at
                    : $order->paid_at;

            $lastRequestAt = DB::table('order_rider_requests')
                ->where('order_id', $order->id)
                ->max('requested_at');

            $enableSendNearby = true;
            $remainingMinutes = 0;

            if ($lastRequestAt) {
                $nextAllowedTime = Carbon::parse($lastRequestAt)->addHours(2);
                $now = now();

                if ($now->lessThan($nextAllowedTime)) {
                    $enableSendNearby = false;
                    $remainingMinutes = $now->diffInMinutes($nextAllowedTime);
                }
            }

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category'    => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand'       => $item->brand,
                        'quantity'    => $item->quantity,
                        'unit'        => $item->quantity_unit,
                        'image'       => $item->image ? asset($item->image) : null
                    ];
                });

            return [
                'order_id' => $order->id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type'  => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'can_send_rider_request' => $enableSendNearby,
                'remaining_minutes'     => $remainingMinutes,

                'door_no'     => $order->door_no,
                'street_name' => $order->street_name,
                'area'        => $order->area,
                'city'        => $order->city,
                'district'    => $order->district_name,
                'state'       => $order->state_name,
                'pincode'     => $order->pincode,

                'rider' => $order->status === 'rider_accepted'
                    ? [
                        'name'   => $order->rider_name,
                        'code'   => $order->rider_code,
                        'mobile' => $order->rider_mobile,
                    ]
                    : null,

                'order_items' => $items
            ];
        });
        
        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getAwaitingRider($orderId)
    {
        $warehouse = auth('warehouse')->user();

        $order = DB::table('orders')
            ->where('id', $orderId)
            ->where('approved_by', $warehouse->id)
            ->where('approved_by_user_type', 'warehouse')
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $riderRequests = DB::table('order_rider_requests as orr')
        ->join('riders as r', 'r.id', '=', 'orr.rider_id')
        ->where('orr.order_id', $orderId)
        ->orderBy('orr.created_at', 'desc')
        ->select(
            'orr.id as request_id',
            'orr.status as request_status',
            'orr.requested_at',
            'orr.distance_km',
            'r.id as rider_id',
            'r.name',
            'r.rider_code',
        )
        ->get()
        ->map(function ($row) {
            return [
                'request_id'   => $row->request_id,
                'status'       => $row->request_status,
                'requested_at' => $row->requested_at,
                'distance_km'  => $row->distance_km,
                'rider' => [
                    'id'     => $row->rider_id,
                    'name'   => $row->name,
                    'code'   => $row->rider_code,
                ]
            ];
        });


        return response()->json([
            'success' => true,
            'order' => [
                'id'     => $order->id,
                'order_no' => $order->order_no,
                'status' => $order->status
            ],
            'count' => $riderRequests->count(),
            'data'  => $riderRequests
        ]);
    }

    public function canResendNearbyRiders($orderId)
    {
        $lastRequest = DB::table('order_rider_requests')
            ->where('order_id', $orderId)
            ->max('requested_at');

        if (!$lastRequest) {
            return response()->json([
                'success' => true,
                'enable_resend' => true,
                'remaining_minutes' => 0
            ]);
        }

        $nextAllowedTime = Carbon::parse($lastRequest)->addHours(2);
        $now = now();

        if ($now->greaterThanOrEqualTo($nextAllowedTime)) {
            return response()->json([
                'success' => true,
                'enable_resend' => true,
                'remaining_minutes' => 0
            ]);
        }

        return response()->json([
            'success' => true,
            'enable_resend' => false,
            'remaining_minutes' => $now->diffInMinutes($nextAllowedTime)
        ]);
    }

    public function getAssignedOrderDetail($orderId)
    {
        $warehouse = auth('warehouse')->user();

        $order = DB::table('orders as o')
            ->leftJoin('order_rider_requests as orr', function ($join) {
                $join->on('orr.order_id', '=', 'o.id')
                    ->where('orr.status', 'active');
            })
            ->leftJoin('riders as r', 'r.id', '=', 'orr.rider_id')
            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')
            ->leftJoin('warehouses as w', 'w.id', '=', 'o.approved_by')
            ->leftJoin('districts as wd', 'wd.id', '=', 'w.district_id')
            ->leftJoin('states as ws', 'ws.id', '=', 'w.state_id')
            ->where('o.id', $orderId)
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.loaded_otp',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                'w.warehouse_name as pickup_name',
                'w.door_no as pickup_door_no',
                'w.street_name as pickup_street',
                'w.area as pickup_area',
                'w.city as pickup_city',
                'wd.name as pickup_district',
                'ws.name as pickup_state',
                'w.pincode as pickup_pincode',
                'w.latitude as pickup_latitude',
                'w.longitude as pickup_longitude',
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Order not found'
            ], 404);
        }

        $products = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $orderId)
            ->select(
                'c.name as category',
                'sc.name as subcategory',
                'b.name as brand',
                'oi.quantity',
                'q.name as quantity_unit',
                'p.image'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'category'    => $item->category,
                    'subcategory' => $item->subcategory,
                    'brand'       => $item->brand,
                    'quantity'    => $item->quantity,
                    'unit'        => $item->quantity_unit,
                    'image'       => $item->image ? asset($item->image) : null
                ];
            });

        $dateTime = $order->payment_method === 'COD'
            ? $order->created_at
            : $order->paid_at;

         $expectedDeliveryDateTime = Carbon::parse($dateTime)->addDays(7);

        return response()->json([
            'success' => true,
            'data' => [
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                'date' => date('Y-m-d', strtotime($dateTime)),
                // 'time' => date('H:i:s', strtotime($dateTime)),
                'expected_delivery_date' => $expectedDeliveryDateTime->format('Y-m-d'),

                'products' => $products,

                'pickup_location' => [
                    'name'     => $order->pickup_name,
                    'door_no'  => $order->pickup_door_no,
                    'street'   => $order->pickup_street,
                    'area'     => $order->pickup_area,
                    'city'     => $order->pickup_city,
                    'district' => $order->pickup_district,
                    'state'    => $order->pickup_state,
                    'pincode'  => $order->pickup_pincode,
                ],

                'drop_location' => [
                    'name'     => $order->drop_name,
                    'door_no'  => $order->drop_door_no,
                    'street'   => $order->drop_street,
                    'area'     => $order->drop_area,
                    'city'     => $order->drop_city,
                    'district' => $order->drop_district,
                    'state'    => $order->drop_state,
                    'pincode'  => $order->drop_pincode,
                ],

                'rider' => $order->rider_name ? [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ] : null,

                'loaded_otp' => $order->loaded_otp ?? null
            ]
        ]);
    }


    public function verifyRiderAndGenerateLoadOtp(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'otp'      => 'required|digits:4'
        ]);

        $warehouse = auth('warehouse')->user();
        $riderRequest = DB::table('order_rider_requests')
            ->where('order_id', $request->order_id)
            ->where('otp', $request->otp)
            ->where('status', 'active')
            ->first();

        if (!$riderRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid rider OTP'
            ], 400);
        }

        DB::beginTransaction();

        try {
            DB::table('order_rider_requests')
                ->where('id', $riderRequest->id)
                ->update([
                    'accepted_at'  => now(),
                    'responded_at' => now(),
                    'updated_at'   => now()
                ]);

            $loadedOtp = rand(1000, 9999);

            DB::table('orders')
                ->where('id', $request->order_id)
                ->update([
                    'status'     => 'rider_verified',
                    'loaded_otp' => $loadedOtp,
                    'loaded_at'  => now(),
                    'updated_at' => now()
                ]);
                
            DB::table('riders')
                ->where('id', $riderRequest->rider_id)
                ->update([
                    'work_status' => 0
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Rider verified & loading OTP generated',
                'data' => [
                    'order_id'   => $request->order_id,
                    'loaded_otp' => $loadedOtp
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong',
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function getLoadedOrders(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        $orders = DB::table('orders as o')
            ->leftJoin('addresses as a', 'o.shipping_address_id', '=', 'a.id')
            ->leftJoin('districts as d', 'd.id', '=', 'a.district_id')
            ->leftJoin('states as s', 's.id', '=', 'a.state_id')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')
            ->select(
                'o.id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.ride_started_at',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                'a.door_no',
                'a.street_name',
                'a.area',
                'a.city',
                'a.pincode',
                'd.name as district_name',
                's.name as state_name'
            )
            ->where('o.user_type', 'store')
            ->where('o.status', 'in_transit')
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->orderBy('o.ride_started_at', 'desc')
            ->get();

        $orders = $orders->map(function ($order) {

            // Use ride_started_at for loaded orders
            $dateTime = $order->ride_started_at ?? $order->paid_at ?? $order->created_at;

            $items = DB::table('order_items as oi')
                ->leftJoin('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as s', 's.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->id)
                ->select(
                    'c.name as category',
                    's.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category'    => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand'       => $item->brand,
                        'quantity'    => $item->quantity,
                        'unit'        => $item->quantity_unit,
                        'image'       => $item->image ? asset($item->image) : null
                    ];
                });

            return [
                'order_id' => $order->id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                'date' => date('Y-m-d', strtotime($dateTime)),
                'time' => date('H:i:s', strtotime($dateTime)),

                'door_no'     => $order->door_no,
                'street_name' => $order->street_name,
                'area'        => $order->area,
                'city'        => $order->city,
                'district'    => $order->district_name,
                'state'       => $order->state_name,
                'pincode'     => $order->pincode,

                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ],

                'order_items' => $items
            ];
        });

        return response()->json([
            'success' => true,
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getLoadedOrderDetail($orderId)
    {
        $warehouse = auth('warehouse')->user();

        $order = DB::table('orders as o')

            /* Rider */
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')

            /* Drop location */
            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')

            /* Pickup location (warehouse) */
            ->leftJoin('warehouses as w', 'w.id', '=', 'o.approved_by')
            ->leftJoin('districts as wd', 'wd.id', '=', 'w.district_id')
            ->leftJoin('states as ws', 'ws.id', '=', 'w.state_id')

            /* Latest Rider Location (SUBQUERY JOIN) */
            ->leftJoin(DB::raw('
                (SELECT rl1.*
                FROM rider_locations rl1
                INNER JOIN (
                    SELECT rider_id, MAX(created_at) AS max_time
                    FROM rider_locations
                    GROUP BY rider_id
                ) rl2
                ON rl1.rider_id = rl2.rider_id 
                AND rl1.created_at = rl2.max_time
                ) as rl
            '), 'rl.rider_id', '=', 'o.rider_id')

            ->where('o.id', $orderId)
            ->where('o.status', 'in_transit')
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')

            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.ride_started_at',

                /* Rider */
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',

                /* Rider live location */
                'rl.latitude as rider_latitude',
                'rl.longitude as rider_longitude',

                /* Pickup (warehouse) */
                'w.warehouse_name as pickup_name',
                'w.door_no as pickup_door_no',
                'w.street_name as pickup_street',
                'w.area as pickup_area',
                'w.city as pickup_city',
                'wd.name as pickup_district',
                'ws.name as pickup_state',
                'w.pincode as pickup_pincode',
                'w.latitude as pickup_latitude',
                'w.longitude as pickup_longitude',

                /* Drop */
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode',
                'da.latitude as drop_latitude',
                'da.longitude as drop_longitude'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Loaded order not found'
            ], 404);
        }

        /* ---------- Products / Materials ---------- */
        $products = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $orderId)
            ->select(
                'c.name as category',
                'sc.name as subcategory',
                'b.name as brand',
                'oi.quantity',
                'q.name as quantity_unit',
                'p.image'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'category'    => $item->category,
                    'subcategory' => $item->subcategory,
                    'brand'       => $item->brand,
                    'quantity'    => $item->quantity,
                    'unit'        => $item->quantity_unit,
                    'image'       => $item->image ? asset($item->image) : null
                ];
            });

        /* ---------- Base Date Logic ---------- */
        $baseDateTime = $order->payment_method === 'COD'
            ? $order->created_at
            : $order->paid_at;

        /* ---------- Expected Delivery Date (+7 days) ---------- */
        $expectedDeliveryDateTime = Carbon::parse($baseDateTime)->addDays(7);

        return response()->json([
            'success' => true,
            'data' => [

                /* Order */
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',

                /* Order Date */
                'order_date' => Carbon::parse($baseDateTime)->format('Y-m-d'),
                // 'order_time' => Carbon::parse($baseDateTime)->format('H:i:s'),

                /* Expected Delivery Date (+7 days) */
                'expected_delivery_date' => $expectedDeliveryDateTime->format('Y-m-d'),
                // 'expected_delivery_time' => $expectedDeliveryDateTime->format('H:i:s'),

                /* Products */
                'products' => $products,

                /* Pickup Location */
                // 'pickup_location' => [
                //     'name'     => $order->pickup_name,
                //     'door_no'  => $order->pickup_door_no,
                //     'street'   => $order->pickup_street,
                //     'area'     => $order->pickup_area,
                //     'city'     => $order->pickup_city,
                //     'district' => $order->pickup_district,
                //     'state'    => $order->pickup_state,
                //     'pincode'  => $order->pickup_pincode,
                //     'latitude' => $order->pickup_latitude,
                //     'longitude'=> $order->pickup_longitude,
                // ],

                /* Drop Location */
                'drop_location' => [
                    'name'     => $order->drop_name,
                    'door_no'  => $order->drop_door_no,
                    'street'   => $order->drop_street,
                    'area'     => $order->drop_area,
                    'city'     => $order->drop_city,
                    'district' => $order->drop_district,
                    'state'    => $order->drop_state,
                    'pincode'  => $order->drop_pincode,
                    // 'latitude' => $order->drop_latitude,
                    // 'longitude'=> $order->drop_longitude,
                ],

                /* Rider */
                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ],

                'tracking' => [
                    'rider_latitude'    => $order->rider_latitude,
                    'rider_longitude'   => $order->rider_longitude,
                    'pickup_latitude'   => $order->pickup_latitude,
                    'pickup_longitude'  => $order->pickup_longitude,
                    'drop_latitude'     => $order->drop_latitude,
                    'drop_longitude'    => $order->drop_longitude,
                ]
            ]
        ]);
    }

    public function getDeliveredOrders(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        $query = DB::table('orders as o')

            /* Rider */
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')

            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')

            ->where('o.status', 'completed')
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')

            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.delivered_at',

                /* Rider */
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',

                /* Drop */
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode'
            );

        if ($request->filled('date')) {
            $query->whereDate('o.delivered_at', $request->date);
        }

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $query->whereBetween(
                'o.delivered_at',
                [
                    Carbon::parse($request->from_date)->startOfDay(),
                    Carbon::parse($request->to_date)->endOfDay()
                ]
            );
        }

        if ($request->filled('month') && $request->filled('year')) {
            $query->whereMonth('o.delivered_at', $request->month)
                ->whereYear('o.delivered_at', $request->year);
        }

        if ($request->filled('year')) {
            $query->whereYear('o.delivered_at', $request->year);
        }

        $orders = $query->orderByDesc('o.delivered_at')->get();

        $orders = $orders->map(function ($order) {

            $deliveredDateTime = $order->delivered_at
                ?? $order->paid_at
                ?? $order->created_at;

            $products = DB::table('order_items as oi')
                ->join('products as p', 'p.id', '=', 'oi.product_id')
                ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
                ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
                ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
                ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
                ->where('oi.order_id', $order->order_id)
                ->select(
                    'c.name as category',
                    'sc.name as subcategory',
                    'b.name as brand',
                    'oi.quantity',
                    'q.name as quantity_unit',
                    'p.image'
                )
                ->get()
                ->map(function ($item) {
                    return [
                        'category'    => $item->category,
                        'subcategory' => $item->subcategory,
                        'brand'       => $item->brand,
                        'quantity'    => $item->quantity,
                        'unit'        => $item->quantity_unit,
                    ];
                });

            return [
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'delivered_date' => Carbon::parse($deliveredDateTime)->format('Y-m-d'),
                'delivered_time' => Carbon::parse($deliveredDateTime)->format('H:i:s'),
                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ],

                'materials' => $products
            ];
        });

        return response()->json([
            'success' => true,
            'filters' => [
                'date'      => $request->date ?? null,
                'from_date' => $request->from_date ?? null,
                'to_date'   => $request->to_date ?? null,
                'month'     => $request->month ?? null,
                'year'      => $request->year ?? null,
            ],
            'count'   => $orders->count(),
            'data'    => $orders
        ]);
    }

    public function getDeliveredOrderDetail($orderId)
    {
        $warehouse = auth('warehouse')->user();

        $order = DB::table('orders as o')
            ->leftJoin('riders as r', 'r.id', '=', 'o.rider_id')

            ->leftJoin('addresses as da', 'da.id', '=', 'o.shipping_address_id')
            ->leftJoin('districts as dd', 'dd.id', '=', 'da.district_id')
            ->leftJoin('states as ds', 'ds.id', '=', 'da.state_id')

            ->leftJoin('warehouses as w', 'w.id', '=', 'o.approved_by')
            ->leftJoin('districts as wd', 'wd.id', '=', 'w.district_id')
            ->leftJoin('states as ws', 'ws.id', '=', 'w.state_id')

            ->where('o.id', $orderId)
            ->where('o.status', 'completed')
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')

            ->select(
                'o.id as order_id',
                'o.order_no',
                'o.status',
                'o.with_shipping',
                'o.payment_method',
                'o.created_at',
                'o.paid_at',
                'o.delivered_at',
                'r.name as rider_name',
                'r.mobile_number as rider_mobile',
                'r.rider_code',
                'w.warehouse_name as pickup_name',
                'w.door_no as pickup_door_no',
                'w.street_name as pickup_street',
                'w.area as pickup_area',
                'w.city as pickup_city',
                'wd.name as pickup_district',
                'ws.name as pickup_state',
                'w.pincode as pickup_pincode',
                'w.latitude as pickup_latitude',
                'w.longitude as pickup_longitude',
                'da.name as drop_name',
                'da.door_no as drop_door_no',
                'da.street_name as drop_street',
                'da.area as drop_area',
                'da.city as drop_city',
                'dd.name as drop_district',
                'ds.name as drop_state',
                'da.pincode as drop_pincode',
                'da.latitude as drop_latitude',
                'da.longitude as drop_longitude'
            )
            ->first();

        if (!$order) {
            return response()->json([
                'success' => false,
                'message' => 'Delivered order not found'
            ], 404);
        }

        $products = DB::table('order_items as oi')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->leftJoin('categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('subcategories as sc', 'sc.id', '=', 'p.subcategory_id')
            ->leftJoin('brands as b', 'b.id', '=', 'p.brand_id')
            ->leftJoin('quantity_units as q', 'q.id', '=', 'p.quantity_unit_id')
            ->where('oi.order_id', $orderId)
            ->select(
                'c.name as category',
                'sc.name as subcategory',
                'b.name as brand',
                'oi.quantity',
                'q.name as quantity_unit',
                'p.image'
            )
            ->get()
            ->map(function ($item) {
                return [
                    'category'    => $item->category,
                    'subcategory' => $item->subcategory,
                    'brand'       => $item->brand,
                    'quantity'    => $item->quantity,
                    'unit'        => $item->quantity_unit,
                    'image'       => $item->image ? asset($item->image) : null
                ];
            });

        $orderDateTime = $order->payment_method === 'COD'
            ? $order->created_at
            : $order->paid_at;

        $deliveredDateTime = $order->delivered_at ?? now();

        $review = DB::table('product_reviews as pr')
            ->join('customers as c', 'c.id', '=', 'pr.customer_id')
            ->where('pr.order_id', $orderId)
            ->select(
                'c.name as reviewer_name',
                'pr.rating',
                'pr.review',
                'pr.created_at'
            )
            ->first();
        
        $reviewData = null;

        if ($review) {
            $reviewData = [
                'reviewer_name' => $review->reviewer_name,
                'rating'        => $review->rating,
                'review'        => $review->review,
                'date'          => Carbon::parse($review->created_at)->format('d-M-Y'),
            ];
        }

        return response()->json([
            'success' => true,
            'data' => [

                /* Order Info */
                'order_id' => $order->order_id,
                'order_no' => $order->order_no,
                'status'   => $order->status,
                'shipping_type' => $order->with_shipping ? 'With Shipping' : 'Without Shipping',
                'payment_method' => $order->payment_method,
                'order_date' => Carbon::parse($orderDateTime)->format('Y-m-d'),
                'order_time' => Carbon::parse($orderDateTime)->format('H:i:s'),
                'delivered_date' => Carbon::parse($deliveredDateTime)->format('Y-m-d'),
                'delivered_time' => Carbon::parse($deliveredDateTime)->format('H:i:s'),
                'materials' => $products,
                'pickup_location' => [
                    'name'     => $order->pickup_name,
                    'door_no'  => $order->pickup_door_no,
                    'street'   => $order->pickup_street,
                    'area'     => $order->pickup_area,
                    'city'     => $order->pickup_city,
                    'district' => $order->pickup_district,
                    'state'    => $order->pickup_state,
                    'pincode'  => $order->pickup_pincode,
                    'latitude' => $order->pickup_latitude,
                    'longitude'=> $order->pickup_longitude,
                ],

                'drop_location' => [
                    'name'     => $order->drop_name,
                    'door_no'  => $order->drop_door_no,
                    'street'   => $order->drop_street,
                    'area'     => $order->drop_area,
                    'city'     => $order->drop_city,
                    'district' => $order->drop_district,
                    'state'    => $order->drop_state,
                    'pincode'  => $order->drop_pincode,
                    'latitude' => $order->drop_latitude,
                    'longitude'=> $order->drop_longitude,
                ],

                'rider' => [
                    'name'   => $order->rider_name,
                    'code'   => $order->rider_code,
                    'mobile' => $order->rider_mobile,
                ],
                'review' => $reviewData,
            ]
        ]);
    }

    public function warehouseSalesReport(Request $request)
    {
        $warehouse = auth('warehouse')->user();

        if (!$warehouse) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 401);
        }

        // Base query for completed orders
        $query = DB::table('orders as o')
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->where('o.status', 'completed');

        // Filters
        if ($request->filled('date')) {
            $query->whereDate('o.delivered_at', $request->date);
        } elseif ($request->filled('from_date') && $request->filled('to_date')) {
            $query->whereBetween('o.delivered_at', [
                $request->from_date . ' 00:00:00',
                $request->to_date . ' 23:59:59'
            ]);
        } elseif ($request->filled('month') && $request->filled('year')) {
            $query->whereMonth('o.delivered_at', $request->month)
                ->whereYear('o.delivered_at', $request->year);
        } elseif ($request->filled('year')) {
            $query->whereYear('o.delivered_at', $request->year);
        }

        // Base query for in-transit orders (loaded)
        $loadedQuery = DB::table('orders')
            ->where('approved_by', $warehouse->id)
            ->where('approved_by_user_type', 'warehouse')
            ->where('status', 'in_transit');

        if ($request->filled('date')) {
            $loadedQuery->whereDate('updated_at', $request->date);
        } elseif ($request->filled('from_date') && $request->filled('to_date')) {
            $loadedQuery->whereBetween('updated_at', [
                $request->from_date . ' 00:00:00',
                $request->to_date . ' 23:59:59'
            ]);
        } elseif ($request->filled('month') && $request->filled('year')) {
            $loadedQuery->whereMonth('updated_at', $request->month)
                        ->whereYear('updated_at', $request->year);
        } elseif ($request->filled('year')) {
            $loadedQuery->whereYear('updated_at', $request->year);
        }

        // Summary using approved_unit_price
        $summary = (clone $query)
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->select(
                DB::raw('COUNT(DISTINCT o.id) as total_orders'),
                DB::raw('SUM(oi.approved_unit_price * oi.quantity) as total_sale_amount'),
                DB::raw('0 as loaded_orders'), 
                DB::raw('COUNT(DISTINCT o.id) as delivered_orders'),
                DB::raw('SUM(CASE WHEN o.with_shipping = 1 THEN 1 ELSE 0 END) as with_shipping_orders'),
                DB::raw('SUM(CASE WHEN o.with_shipping = 0 THEN 1 ELSE 0 END) as without_shipping_orders')
            )
            ->first();

        $summary->loaded_orders = (clone $loadedQuery)->count();

        // Category summary
        $categorySummary = DB::table('order_items as oi')
            ->join('orders as o', 'o.id', '=', 'oi.order_id')
            ->join('products as p', 'p.id', '=', 'oi.product_id')
            ->join('warehouse_products as wp', function($join) use ($warehouse) {
                $join->on('wp.product_id', '=', 'p.id')
                    ->where('wp.warehouse_id', $warehouse->id);
            })
            ->join('categories as c', 'c.id', '=', 'wp.category_id')
            ->where('o.approved_by', $warehouse->id)
            ->where('o.approved_by_user_type', 'warehouse')
            ->where('o.status', 'completed')
            ->when($request->filled('date'), fn($q) => $q->whereDate('o.delivered_at', $request->date))
            ->when($request->filled('from_date') && $request->filled('to_date'), 
                fn($q) => $q->whereBetween('o.delivered_at', [$request->from_date . ' 00:00:00', $request->to_date . ' 23:59:59']))
            ->when($request->filled('month') && $request->filled('year'), 
                fn($q) => $q->whereMonth('o.delivered_at', $request->month)->whereYear('o.delivered_at', $request->year))
            ->when($request->filled('year') && !$request->filled('month'), 
                fn($q) => $q->whereYear('o.delivered_at', $request->year))
            ->select(
                'c.id as category_id',
                'c.name as category_name',
                'c.image as category_image',
                'p.quantity_unit as unit',
                DB::raw('SUM(oi.quantity) as total_quantity'),
                DB::raw('SUM(oi.approved_unit_price * oi.quantity) as total_amount')
            )
            ->groupBy('c.id', 'p.quantity_unit')
            ->orderBy('c.name')
            ->get()
            ->map(function ($item) {
                return [
                    'category_id'    => $item->category_id,
                    'category_name'  => $item->category_name,
                    'category_image' => $item->category_image ? asset($item->category_image) : null,
                    'unit'           => $item->unit,
                    'total_quantity' => $item->total_quantity,
                    'total_amount'   => (float) $item->total_amount
                ];
            });

        // Payment summary using approved_unit_price
        $paymentSummary = (clone $query)
            ->join('order_items as oi', 'oi.order_id', '=', 'o.id')
            ->select(
                DB::raw("SUM(CASE WHEN o.payment_method = 'COD' THEN (oi.approved_unit_price * oi.quantity) ELSE 0 END) as cod_amount"),
                DB::raw("SUM(CASE WHEN o.payment_method IN ('Razorpay','Cashfree') THEN (oi.approved_unit_price * oi.quantity) ELSE 0 END) as online_amount")
            )
            ->first();

        // Filter text
        if ($request->filled('from_date') && $request->filled('to_date')) {
            $filterContent = 'Sales Report from ' . date('d-M-Y', strtotime($request->from_date))
                        . ' to ' . date('d-M-Y', strtotime($request->to_date));
        } elseif ($request->filled('date')) {
            $filterContent = 'Sales Report for ' . date('d-M-Y', strtotime($request->date));
        } elseif ($request->filled('month') && $request->filled('year')) {
            $filterContent = 'Sales Report for ' . date('F', mktime(0,0,0,$request->month,1)) . ' ' . $request->year;
        } elseif ($request->filled('year')) {
            $filterContent = 'Sales Report for Year ' . $request->year;
        } else {
            $filterContent = 'Sales Report';
        }

        return response()->json([
            'success' => true,
            'filter_content' => $filterContent,
            'summary' => [
                'total_sale_amount'        => (float) ($summary->total_sale_amount ?? 0),
                'total_orders'             => (int) ($summary->total_orders ?? 0),
                'loaded_orders'            => (int) ($summary->loaded_orders ?? 0),
                'delivered_orders'         => (int) ($summary->delivered_orders ?? 0),
                'with_shipping_orders'     => (int) ($summary->with_shipping_orders ?? 0),
                'without_shipping_orders'  => (int) ($summary->without_shipping_orders ?? 0)
            ],
            'category_summary' => $categorySummary,
            'payment_summary' => [
                'cod_amount'    => (float) ($paymentSummary->cod_amount ?? 0),
                'online_amount' => (float) ($paymentSummary->online_amount ?? 0),
            ]
        ]);
    }



}
