<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Order extends Model
{
    use HasFactory;

    protected $table = 'orders';

    protected $fillable = [
        'order_no',
        'invoice_no',
        'user_type',
        'user_id',
        'with_shipping',
        'rider_id',
        'shipping_address_id',
        'status',

        'items_total',
        'delivery_fee',
        'grand_total',
        'paid_pending',
        'cancellation_charge',
        'cancelled_at',
        'discount_amount',

        'payment_method',
        'payment_status',
        'payment_response',
        'gateway_order_id',
        'gateway_order_link',
        'cf_link_id',
        'gateway_payment_id',
        'gateway_signature',

        'paid_at',
        'failed_at',
        'payment_failure_reason',

        'approved_by',
        'approved_by_user_type',
        'approved_at',

        'rejected_by',
        'rejected_by_user_type',
        'rejected_at',

        'loaded_otp',
        'loaded_at',

        'ride_started_at',
        'delivery_otp',
        'delivered_at',
    ];


    protected $casts = [
        'with_shipping' => 'boolean',

        'items_total'   => 'decimal:2',
        'delivery_fee'  => 'decimal:2',
        'grand_total'   => 'decimal:2',
        'discount_amount' => 'decimal:2',

        'paid_at'        => 'datetime',
        'failed_at'      => 'datetime',
        'approved_at'    => 'datetime',
        'rejected_at'    => 'datetime',
        'loaded_at'      => 'datetime',
        'ride_started_at'=> 'datetime',
        'delivered_at'   => 'datetime',
        'payment_response' => 'array',
    ];

    public function items()
    {
        return $this->hasMany(OrderItem::class, 'order_id');
    }

    public function seller()
    {
        if ($this->approved_by_user_type === 'warehouse') {
            return $this->belongsTo(Warehouse::class, 'approved_by');
        }

        if ($this->approved_by_user_type === 'store') {
            return $this->belongsTo(Store::class, 'approved_by');
        }

        return null;
    }

    public function sellerName()
    {
        if ($this->approved_by_user_type === 'warehouse' && $this->seller) {
            return $this->seller->warehouse_name;
        }

        if ($this->approved_by_user_type === 'store' && $this->seller) {
            return $this->seller->store_name;
        }

        return 'N/A';
    }

    public function sellerType()
    {
        return $this->approved_by_user_type;
    }

    public function warehouseSeller()
    {
        return $this->belongsTo(Warehouse::class, 'approved_by');
    }

    public function storeSeller()
    {
        return $this->belongsTo(Store::class, 'approved_by');
    }

    public function customerBuyer()
    {
        return $this->belongsTo(Customer::class, 'user_id');
    }

    public function storeBuyer()
    {
        return $this->belongsTo(Store::class, 'user_id');
    }


    public function buyer()
    {
        if ($this->user_type === 'customer') {
            return $this->belongsTo(Customer::class, 'user_id');
        }

        if ($this->user_type === 'store') {
            return $this->belongsTo(Store::class, 'user_id');
        }

        return null;
    }

    public function buyerName()
    {
        if ($this->user_type === 'customer' && $this->buyer) {
            return $this->buyer->name;
        }

        if ($this->user_type === 'store' && $this->buyer) {
            return $this->buyer->store_name;
        }

        return 'N/A';
    }

    public function statusTime()
    {
        switch ($this->status) {

            case 'pending_assignment':
                if (in_array($this->payment_method, ['Razorpay', 'Cashfree'])) {
                    return $this->paid_at;
                }
                return $this->created_at;
        
            case 'order_taken':
                return $this->approved_at;

            case 'waiting_for_rider_approval':
                return null;

            case 'rider_accepted':
                return optional($this->riderRequest)->accepted_at;

            case 'rider_verified':
                return $this->loaded_at;

            case 'in_transit':
                return $this->ride_started_at;

            case 'completed':
                return $this->delivered_at;

            default:
                return null;
        }
    }

    public function shippingLabel()
    {
        return $this->with_shipping ? 'Yes' : 'No';
    }

    public function rider()
    {
        return $this->belongsTo(Rider::class, 'rider_id');
    }

    public function riderRequest()
    {
        return $this->hasOne(OrderRiderRequest::class, 'order_id');
    }

    public function riderRequests()
    {
        return $this->hasMany(OrderRiderRequest::class, 'order_id');
    }

    public function shippingAddress()
    {
        return $this->belongsTo(Address::class, 'shipping_address_id');
    }
}
