<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\DB;

class SellerPayment extends Model
{
    use HasFactory;

    protected $table = 'seller_payments';

    protected $fillable = [
        'order_ids',        // JSON array of order IDs
        'invoice_no',       // unique invoice number
        'seller_id',
        'seller_type',      // warehouse / store
        'amount',           // total approved amount
        'payment_method',   // razorpay / cashfree
        'payment_mode',     // upi / bank / card
        'transaction_id',   // gateway transaction ID
        'payment_response', // raw response from gateway
        'status',           // pending / paid
        'paid_at',
    ];

    protected $casts = [
        'order_ids' => 'array',
        'amount' => 'decimal:2',
        'paid_at' => 'datetime',
    ];

    /**
     * Get the seller (warehouse or store) for this payment.
     */
    public function seller()
    {
        if ($this->seller_type === 'warehouse') {
            return $this->belongsTo(\App\Models\Warehouse::class, 'seller_id');
        }

        if ($this->seller_type === 'store') {
            return $this->belongsTo(\App\Models\Store::class, 'seller_id');
        }

        return null;
    }

    /**
     * Get the orders associated with this payment.
     */
    public function orders()
    {
        return \App\Models\Order::whereIn('id', $this->order_ids)->get();
    }

    /**
     * Mark payment as paid.
     */
    public function markAsPaid($transactionId = null, $paymentResponse = null)
    {
        $this->status = 'paid';
        $this->transaction_id = $transactionId;
        $this->payment_response = $paymentResponse;
        $this->paid_at = now();
        $this->save();
    }

    /**
     * Calculate total amount from order items (approved_unit_price * quantity)
     */
    public static function calculateTotalAmount(array $orderIds)
    {
        return \App\Models\OrderItem::whereIn('order_id', $orderIds)
            ->sum(DB::raw('approved_unit_price * quantity'));
    }

    /**
     * Generate a unique invoice number.
     */
    public static function generateInvoiceNo()
    {
        $prefix = 'INV';
        $date = date('Ymd');
        $count = self::whereDate('created_at', now()->toDateString())->count() + 1;
        return $prefix . $date . str_pad($count, 4, '0', STR_PAD_LEFT);
    }
}
